package com.xebialabs.deployit.plugin.satellite

import com.xebialabs.deployit.engine.tasker.satellite.ActorLocator
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Step, StepExitCode}
import com.xebialabs.deployit.plugin.satellite.extension.{ExtensionsLocator, FileSystemExtensionsLocator}
import com.xebialabs.satellite.future.AwaitForever
import com.xebialabs.satellite.protocol.Directories
import com.xebialabs.xlplatform.satellite.Satellite
import com.xebialabs.xlplatform.settings.CommonSettings
import org.apache.pekko.actor.ActorSystem

import java.net.URI
import scala.beans.BeanProperty

case class CheckExtensionsStep(actorLocator: ActorLocator, @BeanProperty description: String, extensionsLocator: ExtensionsLocator, @transient system: ActorSystem)
                          extends SatelliteActorSystemStep(system) with AwaitForever with SatellitePluginsSugar {

  @BeanProperty val order = Step.DEFAULT_ORDER

  override def execute(ctx: ExecutionContext): StepExitCode = {
    ctx.logOutput(s"Connecting to $actorLocator")
    if (!CommonSettings(satelliteCommunicatorSystem).satellite.enabled) {
      ctx.logError("Satellite has not been enabled on the Deploy server. Please ensure 'deploy.satellite.enabled = true' is set in the deploy-satellite.yaml file of Deploy")
      StepExitCode.FAIL
    } else {

      val installedExtensions: InstalledExtensionsDelta = calculateExtensionsDelta(extensionsLocator, ctx, satelliteCommunicatorSystem)

      printStatus(installedExtensions.plugins, Directories.PLUGINS,  ctx)
      printStatus(installedExtensions.pluginHotfix, Directories.PLUGIN_HOTFIXES, ctx)
      printStatus(installedExtensions.satelliteHotfix, Directories.SATELLITE_HOTFIXES, ctx)
      printStatus(installedExtensions.ext, Directories.EXT, ctx)

      if (installedExtensions.empty) {
        ctx.logOutput(s"Satellite is up-to-date")
        StepExitCode.SUCCESS
      } else {
        ctx.logOutput(s"Found unsynchronized extensions")
        StepExitCode.FAIL
      }
    }
  }

  private def printStatus(opt: Set[URI], extension: String, ctx: ExecutionContext) = {
    if(opt.isEmpty) {
      ctx.logOutput(s"$extension is up-to-date")
    } else {
      ctx.logOutput(s"Need to synchronise $extension")
    }
  }
}

object CheckExtensionsStep {

  def apply(satellite: Satellite)(implicit satelliteCommunicatorSystem: ActorSystem) : CheckExtensionsStep = {
    new CheckExtensionsStep(
      ActorLocator(satellite), s"Check plugins and extensions on satellite ${satellite.getName}",
      FileSystemExtensionsLocator, satelliteCommunicatorSystem
    )
  }

}
