package com.xebialabs.deployit.plugin.satellite

import com.xebialabs.deployit.engine.tasker.satellite.ActorLocator
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Step, StepExitCode}
import com.xebialabs.deployit.plugin.satellite.Pinger.{FIVE_PINGS, isUp}
import com.xebialabs.deployit.plugin.satellite.WaitingForRestartStep.RetryOnceMore
import com.xebialabs.xlplatform.satellite.Satellite
import com.xebialabs.xlplatform.settings.CommonSettings
import org.apache.pekko.actor.ActorSystem

import scala.beans.BeanProperty
import scala.concurrent.duration._
import scala.concurrent.{Await, ExecutionContextExecutor, Future, Promise}
import scala.util.Try

class WaitingForRestartStep(maxAttempts: Int, delay: FiniteDuration, pingTimeout: FiniteDuration,
                            actorLocator: ActorLocator, satelliteAddress: SatelliteAddress,
                            system: ActorSystem) extends SatelliteActorSystemStep(system) {

  @BeanProperty val order: Int = Step.DEFAULT_ORDER
  @BeanProperty val description = "Wait for satellite to restart"

  @transient implicit private lazy val dispatcher: ExecutionContextExecutor = satelliteCommunicatorSystem.dispatcher

  private var remainingAttempt: Int = maxAttempts

  override def execute(ctx: ExecutionContext): StepExitCode = {
    remainingAttempt = maxAttempts

    Try(
      Await.result(satelliteRestarted(ctx), atMost = delay * maxAttempts * 4)
    ).recover { case _ => StepExitCode.FAIL }.get
  }

  private def satelliteRestarted(ctx: ExecutionContext): Future[StepExitCode] = {
    remainingAttempt -= 1

    if (remainingAttempt < 0) {
      return Future.successful(StepExitCode.FAIL)
    }

    val result = Promise[StepExitCode]()

    satelliteCommunicatorSystem.scheduler.scheduleOnce(delay) {
      if (isUp(satelliteAddress, actorLocator, pingTimeout, FIVE_PINGS, Clock())(ctx, satelliteCommunicatorSystem)) {
        result.success(StepExitCode.SUCCESS)
        ctx.logOutput("Restart successful.")
      } else {
        result.failure(RetryOnceMore)
        ctx.logOutput("Retrying")
      }
    }

    result.future.recoverWith {
      case RetryOnceMore => satelliteRestarted(ctx)
    }
  }
}

object WaitingForRestartStep {
  private object RetryOnceMore extends Throwable

  def apply(satellite: Satellite, maxAttempts: Int, delay: Int)
           (implicit satelliteCommunicatorSystem: ActorSystem): WaitingForRestartStep =
  {
    val pingTimeout = CommonSettings(satelliteCommunicatorSystem).satellite.remoteAskTimeout.duration
    apply(satellite, maxAttempts, delay.seconds, pingTimeout, ActorLocator(satellite))
  }

  private[satellite] def apply(satellite: Satellite, maxAttempts: Int, delay: FiniteDuration,
                               pingTimeout: FiniteDuration, actorLocator: ActorLocator)
                              (implicit satelliteCommunicatorSystem: ActorSystem): WaitingForRestartStep =
    new WaitingForRestartStep(maxAttempts, delay, pingTimeout, actorLocator, SatelliteAddress(satellite), satelliteCommunicatorSystem)
}