package com.xebialabs.deployit.plugin.satellite

import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.plugin.api.udm.Parameters
import com.xebialabs.deployit.tasksystem.TaskActorSystem
import com.xebialabs.xlplatform.satellite.{Satellite, SatelliteGroup}
import org.apache.pekko.actor.{ActorRef, ActorSystem}

import java.util.{List => JList}
import scala.jdk.CollectionConverters._

object SatelliteTasksHelper {

  def ping(satellite: Satellite): JList[Step] = {
    implicit val satelliteCommunicatorSystem: ActorSystem = TaskActorSystem.actorSystem
    satellite match {
      case group: SatelliteGroup => group.getSatellites.asScala.toList.map(pingSat).asJava
      case sat: Satellite => (pingSat(sat) :: Nil).asJava
    }
  }

  def sync(satellite: Satellite, parameters: Parameters): JList[Step] = {
    implicit val satelliteCommunicatorSystem: ActorSystem = TaskActorSystem.actorSystem
    implicit val uploader: ActorRef = SatelliteActors.uploader

    satellite match {
      case group: SatelliteGroup =>
        val sats = group.getSatellites.asScala.toList
        (sats.flatMap(syncRestart) ::: sats.map(waitFor(_, parameters))).asJava
      case sat: Satellite => (syncRestart(sat) ::: (waitFor(sat, parameters) :: Nil)).asJava
    }
  }

  def restartSatellite(satellite: Satellite, parameters: Parameters): JList[Step] = {
    implicit val satelliteCommunicatorSystem: ActorSystem = TaskActorSystem.actorSystem
    satellite match {
      case group: SatelliteGroup =>
        val sats = group.getSatellites.asScala.toList
        (sats.flatMap(restart(_, parameters)) ::: sats.map(waitFor(_, parameters))).asJava
      case sat: Satellite => (restart(sat, parameters) ::: (waitFor(sat, parameters) :: Nil)).asJava
    }
  }

  private[this] def syncRestart(sat: Satellite)(implicit satelliteCommunicatorSystem: ActorSystem, uploader: ActorRef): List[Step] =
    SyncExtensionsStep(sat) :: SoftRestartSatelliteStep(sat) :: Nil

  private[this] def restart(sat: Satellite, parameters: Parameters)(implicit satelliteCommunicatorSystem: ActorSystem): List[Step] =
    if (parameters.getProperty("force")) ForceRestartSatelliteStep(sat) :: Nil else SoftRestartSatelliteStep(sat) :: Nil

  private[this] def waitFor(sat: Satellite, parameters: Parameters)(implicit satelliteCommunicatorSystem: ActorSystem): Step =
    WaitingForRestartStep(sat, parameters.getProperty("maxAttempts"), parameters.getProperty("delay"))

  private[this] def pingSat(sat: Satellite)(implicit system: ActorSystem): Step = PingSatelliteStep(sat)

}
