package com.xebialabs.deployit.plugin.satellite

import com.xebialabs.deployit.engine.tasker.satellite.ActorLocator
import com.xebialabs.deployit.plugin.api.flow.ExecutionContext
import com.xebialabs.deployit.plugin.satellite.Pinger.{Unavailable, Up}
import com.xebialabs.satellite.protocol.{Paths, Ping, PingReply}
import com.xebialabs.xlplatform.satellite.Satellite
import com.xebialabs.xlplatform.settings.CommonSettings
import org.apache.pekko.actor._
import org.apache.pekko.pattern.ask
import org.apache.pekko.util.Timeout

import scala.concurrent.duration._
import scala.concurrent.{Await, ExecutionContextExecutor}

object Pinger {

  val FIVE_PINGS = 5

  def isUp(satellite: Satellite, noPings: Int = FIVE_PINGS)
          (implicit ctx: ExecutionContext, satelliteCommunicatorSystem: ActorSystem): Boolean =
    isUp(SatelliteAddress(satellite), ActorLocator(satellite), defaultPingTimeout(satelliteCommunicatorSystem), noPings, Clock())

  def defaultPingTimeout(satelliteCommunicatorSystem: ActorSystem): FiniteDuration =
    CommonSettings(satelliteCommunicatorSystem).satellite.pingTimeout.duration

  def isUp(satelliteAddress: SatelliteAddress, actorLocator: ActorLocator, pingTimeout: FiniteDuration,
           pingCount: Int, clock: Clock)
          (implicit ctx: ExecutionContext, satelliteCommunicatorSystem: ActorSystem): Boolean =
  {
    val grace = 1.second
    implicit val timeout: Timeout = Timeout(pingTimeout + grace) // allow Pinger to send back a result even after timeout
    implicit val dispatcher: ExecutionContextExecutor = satelliteCommunicatorSystem.dispatcher

    ctx.logOutput(s"Connecting to satellite at $satelliteAddress")
    val pinger = satelliteCommunicatorSystem.actorOf(Pinger.props(ctx, pingTimeout, satelliteAddress, clock, pingCount))
    val pingFuture = (pinger ? Pinger.Start(actorLocator.locate(Paths.ping))).mapTo[PingResult]
      .recover { case error =>
        ctx.logError(s"Operation failed  (${error.getMessage})")
        Unavailable
      }
    Await.result(pingFuture, timeout.duration + grace) == Up // more grace time allow local handling even after timeout
  }

  def props(ctx: ExecutionContext, timeout: FiniteDuration, satelliteAddress: SatelliteAddress, clock: Clock = Clock(),
            noPings: Int = Pinger.FIVE_PINGS) =
    Props(new Pinger(ctx, timeout, satelliteAddress, clock, noPings))

  sealed trait PingResult

  case class Start(target: ActorSelection)

  case object Up extends PingResult

  case object Unavailable extends PingResult

}

class Pinger(ctx: ExecutionContext, pingTimeout: FiniteDuration, satelliteAddress: SatelliteAddress, clock: Clock, pingCount: Int) extends Actor {

  def receive: Receive = if (!CommonSettings(context.system).satellite.enabled) notEnabled else waitingToStart

  def notEnabled: Receive = {
    case Pinger.Start(_) =>
      ctx.logError("Satellite has not been enabled on the Deploy server. " +
        "Please ensure 'deploy.satellite.enabled = true' is set in the deploy-satellite.yaml file of Deploy")
      sender() ! Unavailable
      context.stop(self)
  }

  def waitingToStart: Receive = {
    case Pinger.Start(target) =>
      val startTime = clock.currentTimeMillis()
      target ! Ping
      ctx.logOutput(s"Waiting max $pingTimeout for connection to satellite at $satelliteAddress")
      context.setReceiveTimeout(pingTimeout)
      context become awaitPings(target, startTime, pingCount, sender())
  }

  def awaitPings(target: ActorSelection, startTime: Long, remainingPingsToReceive: Int, origRequester: ActorRef): Receive = {
    case PingReply(uptime) =>
      val stopTime = clock.currentTimeMillis()
      if (remainingPingsToReceive == pingCount) {
        ctx.logOutput(s"Satellite is live (uptime: $uptime sec)")
      }
      ctx.logOutput(s"Ping: ${stopTime - startTime} ms")

      if (remainingPingsToReceive > 1) {
        val newStartTime = clock.currentTimeMillis()
        target ! Ping
        context become awaitPings(target, newStartTime, remainingPingsToReceive - 1, origRequester)
      } else {
        origRequester ! Up
        context stop self
      }
    case ReceiveTimeout =>
      ctx.logError(s"The satellite at $satelliteAddress cannot be reached within $pingTimeout. Please check whether it is running.")
      origRequester ! Unavailable
      context stop self
  }
}
