package com.xebialabs.deployit.plugin.satellite

import akka.actor.ActorSystem
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, StepExitCode}
import com.xebialabs.xlplatform.satellite.{Satellite, SatelliteGroup}

import scala.jdk.CollectionConverters._
import scala.util.Random

class ChooseSatelliteInGroupStep(satelliteGroup: SatelliteGroup, system: ActorSystem) extends SatelliteActorSystemStep(system) {
  /**
    * The order of the step in the step list.
    *
    * @return the order.
    */
  override def getOrder: Int = 10

  /**
    * Returns a description of what this step will really do on execution.
    *
    * @return the description.
    */
  override def getDescription: String = s"Choose a satellite to execute task in $satelliteGroup"

  /**
    * @param ctx Context info required to execute the step.
    * @return exit code of the execution, successful, failed or paused.
    * @throws Exception Any type of exception can be thrown,
    *                   this to prevent implementors of having to wrap any non-runtime exceptions.
    */
  override def execute(ctx: ExecutionContext): StepExitCode = {
    val chosen = new RandomSatelliteChooser(ctx, satelliteCommunicatorSystem).pick(satelliteGroup.getSatellites.asScala.toList.sortBy(_.getId))
    chosen.map({ sat =>
      ctx.logOutput(s"Chose Satellite $sat for group $satelliteGroup")
      satelliteGroup.setChosenSatellite(sat)
      StepExitCode.SUCCESS
    }).getOrElse({
      ctx.logError(s"There are no active satellites in group $satelliteGroup. Please assign at least one active satellite and restart the task.")
      StepExitCode.FAIL
    })
  }
}

object ChooseSatelliteInGroupStep {

  def apply(satelliteGroup: SatelliteGroup)(implicit satelliteCommunicatorSystem: ActorSystem): ChooseSatelliteInGroupStep =
    new ChooseSatelliteInGroupStep(satelliteGroup, satelliteCommunicatorSystem)
}

trait SatelliteChooser {
  def pick(satellites: List[Satellite]): Option[Satellite]
}

class RandomSatelliteChooser(ctx: ExecutionContext, satelliteCommunicatorSystem: ActorSystem) extends SatelliteChooser {
  override def pick(satellites: List[Satellite]): Option[Satellite] = {
    val randomized = Random.shuffle(satellites)
    ctx.logOutput(s"Trying satellite group ${satellites.mkString("[", ", ", "]")} in order ${randomized.mkString(", ")}")
    randomized.find {
      case sat if Pinger.isUp(sat, 1)(ctx, satelliteCommunicatorSystem) => true
      case sat =>
        ctx.logOutput(s"Satellite $sat is not responding to pings, not selecting it.")
        false
    }
  }
}
