package com.xebialabs.deployit.plugin.satellite

import akka.actor.{ActorRef, ActorSelection, ActorSystem, ExtendedActorSystem}
import akka.pattern.ask
import akka.util.Timeout
import com.xebialabs.deployit.engine.tasker
import com.xebialabs.deployit.engine.tasker.StateChangeEventListenerActor.TaskStateEvent
import com.xebialabs.deployit.engine.tasker._
import com.xebialabs.deployit.engine.tasker.satellite.ActorLocator
import com.xebialabs.deployit.io.ArtifactFile
import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, Step, StepExitCode}
import com.xebialabs.deployit.plugin.satellite.UploadTaskSupervisor.Protocol.{FileToUpload, UploadConfig, UploadFilesTask, UploadTaskFinished}
import com.xebialabs.satellite.future.AwaitForever
import com.xebialabs.satellite.protocol._
import com.xebialabs.satellite.serialization.{FilesToUpload, TaskSpecificationSerialization, TaskSpecificationSerializationExtension}
import com.xebialabs.xlplatform.satellite.Satellite

import scala.beans.BeanProperty
import scala.concurrent.ExecutionContextExecutor
import scala.concurrent.duration._
import scala.util.{Failure, Success, Try}

case class SendToSatelliteStep(satellite: Satellite, actorLocator: ActorLocator, @BeanProperty description: String, cleanupCache: TaskOnSatelliteCleanupCache,
                               @transient system: ActorSystem, @transient _uploader: ActorRef)
  extends SatelliteActorSystemStep(system) with AwaitForever {

  override def execute(ctx: ExecutionContext): StepExitCode = {
    val taskPreparationStep = prepareTask(ctx)

    val registration = taskPreparationStep.map(registerTask(_, ctx))

    registration match {
      case Failure(exception) =>
        ctx.logError(exception.getMessage)
        StepExitCode.FAIL

      case Success(value) =>
        value
    }
  }

  def prepareTask(ctx: ExecutionContext): Try[TaskSpecificationExchange] = {
    val task = ctx.getAttribute(TaskExecutionContext.CACHE_KEY).asInstanceOf[TaskSpecification]
    val taskSerializer: TaskSpecificationSerialization = TaskSpecificationSerializationExtension(satelliteCommunicatorSystem).createSerializer()
    taskSerializer.toBinary(task).flatMap {
      case (bytes, files, artifactClass, stepTypes) =>
        sendFiles(task.getId, ctx, files) match {
          case UploadTaskFinished(_, Some(error)) =>
            Failure(new Throwable(s"Error while sending file to satellite. ($error)"))
          case _ =>
            Success(TaskSpecificationExchange(stepTypes.map(_.getName), bytes, task.getId, files.map(_.getId), artifactClass.map(_.getName)))
        }
    }
  }

  def registerTask(taskPreparationStep: TaskSpecificationExchange, ctx: ExecutionContext): StepExitCode = {
    implicit val timeout = Timeout(1.day)
    implicit val executionContext = satelliteCommunicatorSystem.dispatcher

    val remoteTaskActor = actorLocator.locate(tasker.satellite.Paths.tasks)(satelliteCommunicatorSystem)

    val sendingTask = (remoteTaskActor ? taskPreparationStep).mapTo[TaskReply].map {
      case Registered =>
        registerTaskEventForwarder(taskPreparationStep.taskId, remoteTaskActor)
        cleanupCache.registerForCleanup(satellite)
        StepExitCode.SUCCESS

      case RegistrationFailedForMissingStep(missingSteps) =>
        ctx.logError(s"Steps of types '${missingSteps.mkString(", ")}' are missing on the satellite. Please keep your plugins in sync.")
        StepExitCode.FAIL

      case RegistrationFailedForUnexpectedError(e) =>
        ctx.logError("Satellite can't deserialize task specification. Please keep your plugins in sync.", e)
        StepExitCode.FAIL

      case RegistrationFailedForMissingFiles(missingFiles) =>
        ctx.logError(s"Files '${missingFiles.mkString(", ")}' are missing on the satellite. Please retry.")
        StepExitCode.FAIL

      case RegistrationFailedForMissingArtifactsImplem(missingArtifactTypes) =>
        ctx.logError(s"Artifact of types '${missingArtifactTypes.mkString(", ")}' are missing on the satellite. Please keep your plugins in sync.")
        StepExitCode.FAIL

      case AlreadyRegistered =>
        ctx.logError("Task is already registered")
        cleanupCache.registerForCleanup(satellite)
        StepExitCode.FAIL
    }

    blockOrThrow(sendingTask)
  }

  def registerTaskEventForwarder(id: TaskId, remoteTaskActor: ActorSelection): Boolean = {
    val taskEventForwarder: ActorRef = satelliteCommunicatorSystem.actorOf(TaskEventForwarder.props(id, remoteTaskActor), s"TaskEventForwarder-$id-${SatelliteAddress(satellite)}")
    satelliteCommunicatorSystem.eventStream.subscribe(taskEventForwarder, classOf[TaskStateEvent])
  }

  def sendFiles(taskId: TaskId, ctx: ExecutionContext, files: FilesToUpload) = {
    implicit val timeout: Timeout = Timeout(1.day)
    implicit val executionContext: ExecutionContextExecutor = satelliteCommunicatorSystem.dispatcher

    val filesToUpload = files.map(artifact => FileToUpload(artifact.getId, artifact.getFile.asInstanceOf[ArtifactFile]))
    val config = UploadConfig(taskId, ctx, actorLocator, SatelliteAddress(satellite))

    blockOrThrow(uploader ? UploadFilesTask(config, filesToUpload))
  }

  @transient lazy val uploader: ActorRef = if (_uploader != null) _uploader else SatelliteActors.uploader

  @BeanProperty val order = Step.DEFAULT_ORDER
}

object SendToSatelliteStep {
  def apply(satellite: Satellite, cleanupCache: TaskOnSatelliteCleanupCache)(implicit satelliteCommunicatorSystem: ActorSystem, uploader: ActorRef): SendToSatelliteStep =
    new SendToSatelliteStep(satellite, ActorLocator(satellite), s"Send task to ${satellite.getName}", cleanupCache, satelliteCommunicatorSystem, uploader)
}
