package com.xebialabs.satellite.streaming

import java.nio.ByteBuffer
import java.nio.channels.{Channels, WritableByteChannel}
import java.nio.file.{Files, Path, StandardOpenOption}

import grizzled.slf4j.Logging

trait ChannelFactory {

  def createFileChannel(path: Path): WritableByteChannel

}

object ChannelFactory {

  object Nio extends ChannelFactory {

    def createFileChannel(path: Path): WritableByteChannel = {
      path.toFile.getParentFile.mkdirs()

      val targetPath = if (path.toFile.exists()) path else Files.createFile(path)

      val newOutputStream = Files.newOutputStream(targetPath, StandardOpenOption.CREATE, StandardOpenOption.WRITE)

      new LoggingWritableByteChannel(path, Channels.newChannel(newOutputStream))
    }

    private class LoggingWritableByteChannel(path: Path, channel: WritableByteChannel) extends WritableByteChannel with Logging {

      override def write(src: ByteBuffer): Int = logAction("write") {
        channel.write(src)
      }

      override def isOpen: Boolean = logAction("isOpen") {
        channel.isOpen
      }

      override def close(): Unit = logAction("close") {
        channel.close()
      }

      private def logAction[U](actionName: String)(action: => U): U = {
        try {
          trace(s"Calling $actionName for $path")
          action
        } catch {
          case t: Throwable =>
            error(s"Error while $actionName", t)
            throw t
        }
      }
    }

  }

}
