package com.xebialabs.satellite.streaming

import akka.NotUsed
import akka.stream.scaladsl.Flow
import akka.stream.stage._
import akka.stream.{Attributes, FlowShape, Inlet, Outlet}
import akka.util.ByteString
import com.xebialabs.deployit.checksum.DefaultChecksumAlgorithmProviderFactory.defaultComparisonAlgorithm
import jakarta.xml.bind.annotation.adapters.HexBinaryAdapter

object DigesterStage {
  type ProcessDigest = Digest => Unit

  def flow(onComplete: ProcessDigest): Flow[ByteString, ByteString, NotUsed] = Flow[ByteString].via(new StreamDigester(onComplete))

  private class StreamDigester(onComplete: ProcessDigest) extends GraphStage[FlowShape[ByteString, ByteString]] {

    val in: Inlet[ByteString] = Inlet[ByteString]("StreamDigester.in")
    val out: Outlet[ByteString] = Outlet[ByteString]("StreamDigester.out")

    override def shape: FlowShape[ByteString, ByteString] = FlowShape.of(in, out)

    private val digester = new Digester

    override def createLogic(inheritedAttributes: Attributes): GraphStageLogic = new GraphStageLogic(shape) {

      setHandler(in, new InHandler {
        override def onPush(): Unit = {
          val input: ByteString = grab(in)
          val bytes: Array[Byte] = new Array[Byte](input.length)
          input.copyToArray(bytes)
          digester.update(bytes)
          push(out, input)
        }

        override def onUpstreamFinish(): Unit = {
          onComplete(digester.digest())
          super.onUpstreamFinish()
        }
      })

      setHandler(out, new OutHandler {
        override def onPull(): Unit = {
          pull(in)
        }
      })

    }
  }

  case class Digest(checksum: String, size: Long)

  private[streaming] class Digester {
    private val adapter = new HexBinaryAdapter()
    private val digester = defaultComparisonAlgorithm.getMessageDigest
    var size = 0L

    def update(bytes: Array[Byte]): Unit = {
      size += bytes.length
      digester.update(bytes)
    }

    def digest(): Digest = Digest(adapter.marshal(digester.digest()), size)
  }

}

