package com.xebialabs.satellite.serialization.kryo

import com.esotericsoftware.kryo.io.{Input, Output}
import com.esotericsoftware.kryo.{Kryo, Serializer}
import com.xebialabs.deployit.io.{ArtifactAwareFile, DerivedArtifactFile}
import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.plugin.api.udm.artifact.{Artifact, DerivedArtifact, SourceArtifact}
import com.xebialabs.overthere.OverthereFile
import com.xebialabs.xltype.serialization.SerializationException

class StepSerializer(kryo: Kryo, val typeClass: Class[_ <: Step]) extends BaseCustomSerializer[Any](kryo, typeClass) {

  override def write(kryo: Kryo, output: Output, obj: Any): Unit = {
    getFields.foreach { cachedField =>
      val field = cachedField.getField

      if (!classOf[OverthereFile].isAssignableFrom(field.getType)) {
        cachedField.write(output, obj)
      } else {
        field.setAccessible(true)
        val file = field.get(obj)

        if (file != null) {
          file match {
            case artifactFile: ArtifactAwareFile =>
              val underlyingArtifact = artifactFile.underlyingArtifact()

              val registration = kryo.writeClass(output, underlyingArtifact.getClass)

              val serializer = registration.getSerializer.asInstanceOf[Serializer[Artifact]]

              kryo.writeObject(output, underlyingArtifact, serializer)

            case _ =>
              throw new SerializationException(s"field [$field] of class [${obj.getClass.getName}] is a file on a step which is not proxied to be serialized.")
          }
        } else {
          kryo.writeClass(output, classOf[Object])
          output.writeByte(Kryo.NULL)
        }
      }
    }
  }

  override def read(kryo: Kryo, input: Input, clazz: Class[Any]): Any = {
    val obj = create(kryo, input, clazz)

    kryo.reference(obj)

    getFields.foreach(cachedField => {
      val field = cachedField.getField
      if (!classOf[OverthereFile].isAssignableFrom(field.getType)) {
        cachedField.read(input, obj)
      } else {
        val registration = kryo.readClass(input)

        val serializer = registration
          .getSerializer
          .asInstanceOf[Serializer[Artifact]]

        val artifact = kryo.readObjectOrNull(input, registration.getType, serializer)

        if (artifact != null) {
          artifact match {
            case sourceArtifact: SourceArtifact =>
              field.set(obj, sourceArtifact.getFile)
            case derivedArtifact: DerivedArtifact[_] =>
              field.set(obj, new DerivedArtifactFile(derivedArtifact))
            case _ =>
          }
        }
      }
    })

    obj
  }
}
