package com.xebialabs.satellite.serialization.kryo

import ai.digital.deploy.task.serdes.kryo.KryoSerializer
import akka.actor.ExtendedActorSystem
import com.esotericsoftware.kryo.Kryo
import com.esotericsoftware.kryo.io.{Input, Output}
import com.xebialabs.deployit.engine.spi.execution.ExecutionStateListener
import com.xebialabs.deployit.engine.tasker.{StepBlock, TaskSpecification}
import com.xebialabs.deployit.plugin.api.flow.Step
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact
import com.xebialabs.satellite.serialization.TaskSpecificationSerialization.{artifactTypes, stepTypes}
import com.xebialabs.satellite.serialization.{FilesCache, FilesToUpload, TaskSpecificationSerialization, TaskSpecificationSerializationProvider, UploadedFiles}

import java.io.ByteArrayOutputStream
import scala.util.Try

object KryoTaskSpecificationSerialization extends TaskSpecificationSerializationProvider {
  def apply(extendedActorSystem: ExtendedActorSystem): TaskSpecificationSerialization =
    new KryoTaskSpecificationSerialization(extendedActorSystem)
}

private class KryoTaskSpecificationSerialization(val system: ExtendedActorSystem) extends TaskSpecificationSerialization {

  override def fromBinary(bytes: Array[Byte],
                          files: UploadedFiles,
                          artifactImplems: Seq[Class[_ <: Artifact]],
                          stepImplems: Seq[Class[_ <: Step]]): Try[TaskSpecification] =
    Try {
      val kryo = createKryo()

      artifactImplems.foreach(clazz => kryo.register(clazz, new ArtifactSerializer(kryo, clazz, Right(files))))
      stepImplems.foreach(clazz => kryo.register(clazz, new StepSerializer(kryo, clazz)))

      val input: Input = new Input(bytes)
      val deserialized = kryo.readObject(input, classOf[TaskSpecification])

      input.close()
      deserialized
    }

  override def toBinary(obj: TaskSpecification): Try[(Array[Byte], FilesToUpload, Seq[Class[_ <: Artifact]], Seq[Class[_ <: Step]])] = {
    val files = FilesCache.empty

    val kryo = createKryo()

    artifactTypes.foreach(clazz => kryo.register(clazz, new ArtifactSerializer(kryo, clazz, Left(files))))
    stepTypes.foreach(clazz => kryo.register(clazz, new StepSerializer(kryo, clazz)))

    Try {
      val outputStream = new ByteArrayOutputStream()
      val output = new Output(outputStream)
      kryo.writeObject(output, obj)

      output.flush()
      output.close()

      (outputStream.toByteArray, files.toSet, artifactTypes, stepTypes)
    }
  }

  private def createKryo(): Kryo = {
    val kryo = KryoSerializer.getMagicCombination(system)

    kryo.register(classOf[TaskSpecification], new TaskSpecificationSerializer(kryo))
    kryo.register(classOf[StepBlock], new StepBlockSerializer(kryo))
    kryo.addDefaultSerializer(classOf[ExecutionStateListener], new ExecutionStateListenerSerializer(kryo))

    kryo
  }
}
