package com.xebialabs.deployit.inspection

import java.lang.reflect.{InvocationTargetException, Method}

import com.xebialabs.deployit.plugin.api.inspection.{Inspect, InspectionContext, InspectionPlanningContext}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import grizzled.slf4j.Logging

import scala.jdk.CollectionConverters._

/**
  * Discovers ConfigurationItems from reality and/or syncs them.
  */
object Inspector extends Logging {

  def inspect(item: ConfigurationItem, context: InspectionContext): Unit = {
    checkRequiredDiscoveryProperties(item)
    val inspectMethod = findInspectMethod(item)
    if (inspectMethod != null) {
      logger.debug(s"Invoking @Inspect method on $item")
      val parameterTypes = inspectMethod.getParameterTypes
      if (parameterTypes(0) == classOf[InspectionPlanningContext]) callInspectionMethod(inspectMethod, item, new InspectionContextAdapter(context))
      else if (parameterTypes(0) == classOf[InspectionContext]) callInspectionMethod(inspectMethod, item, context)
      // DEPLOYITPB-3766 Enabled here to revert to previous situation (see below)
      context.inspected(item)
    }
    // Always call inspected. If there is no inspectMethod, this CI needs no further actions thus it is inspected.
    // DEPLOYITPB-3766 Disabled because it breaks WAS itests
    //        context.inspected(item);
  }

  private def callInspectionMethod(inspectMethod: Method, item: ConfigurationItem, ctx: Any): Unit = {
    try inspectMethod.invoke(item, ctx)
    catch {
      case e@(_: IllegalArgumentException | _: IllegalAccessException) =>
        throw new RuntimeException("Cannot invoke @Inspect method on " + item, e)
      case e: InvocationTargetException =>
        throw new RuntimeException("Error invoking @Inspect method on " + item, e)
    }
  }

  private def checkRequiredDiscoveryProperties(item: ConfigurationItem): Unit = {
    val descriptor = item.getType.getDescriptor
    for (propertyDescriptor <- descriptor.getPropertyDescriptors.asScala) {
      if (propertyDescriptor.isInspectionProperty && propertyDescriptor.isRequiredForInspection) {
        val o = propertyDescriptor.get(item)
        if (o == null) throw new IllegalArgumentException("Missing required property for discovery " + propertyDescriptor.getName)
      }
    }
  }

  private def findInspectMethod(item: ConfigurationItem): Method = {
    val methods = item.getClass.getMethods
    for (method <- methods) {
      if (method.isAnnotationPresent(classOf[Inspect])) return method
    }
    null
  }

}
