package com.xebialabs.deployit.inspection.service.discovery

import ai.digital.deploy.tasker.common.TaskMetadata.TASK_TYPE
import ai.digital.deploy.tasker.common.TaskType
import com.google.common.base.Strings.lenientFormat
import com.xebialabs.deployit.engine.tasker.{BlockBuilders, TaskSpecification}
import com.xebialabs.deployit.inspection.service.discovery.DiscoveryMessages.{DiscoveryResultsFound, RetrieveResults}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.repository.WorkDir
import com.xebialabs.deployit.security.Permissions
import com.xebialabs.deployit.task.TaskMetadataModifier.putMetadata
import com.xebialabs.deployit.task.WorkdirCleanerTrigger
import org.apache.pekko.actor.{ActorRef, ActorSystem}
import org.apache.pekko.pattern._
import org.apache.pekko.util.Timeout

import java.util.concurrent.TimeUnit
import scala.concurrent.Await

class DiscoveryService(val system: ActorSystem, val workerManager: ActorRef, inProcessTaskEngine: Boolean) {

  private val discoveryResultsRetriever = system.actorOf(DiscoveryResultsRetriever.props(workerManager, inProcessTaskEngine), DiscoveryResultsRetriever.name)

  def prepareDiscovery(ci: ConfigurationItem, workDir: WorkDir): TaskSpecification = {
    val inspectionContext = new DiscoveryContext
    inspectionContext.discovered(ci)

    val description = "Inspection of " + ci.getId
    val stepBlock = BlockBuilders.steps(description, None, List()).build()
    val spec = new TaskSpecification(description, Permissions.getAuthentication, workDir, stepBlock, inspectionContext, false)
    spec.getListeners.add(new WorkdirCleanerTrigger(workDir))
    spec.getListeners.add(new DiscoveryTrigger(inspectionContext))
    putMetadata(spec, TASK_TYPE, TaskType.INSPECTION.name)
    spec
  }

  def getDiscoveryResults(taskId: String): DiscoveryResult = {
    implicit val timeout: Timeout = Timeout(10, TimeUnit.SECONDS)
    val discoveryResult = try {
      Await.result(discoveryResultsRetriever ? RetrieveResults(taskId), timeout.duration)
        .asInstanceOf[DiscoveryResultsFound]
        .discoveryResult
    } catch {
      case _: Throwable => throw new IllegalArgumentException(lenientFormat("Could not find any results associated with the id [%s]", taskId));
    }
    discoveryResult
      .getOrElse(throw new IllegalArgumentException(lenientFormat("Could not find any results associated with the id [%s]", taskId)))
  }
}
