package com.xebialabs.deployit.inspection.service.discovery

import akka.actor.{Actor, ActorRef, ActorSystem, Cancellable, Props}
import com.xebialabs.deployit.engine.tasker.TaskId
import com.xebialabs.deployit.engine.tasker.distribution.WorkerManager.messages.{Found, Publish}
import com.xebialabs.deployit.engine.tasker.distribution.{FindOneAggregator, MessageSequenceReceiver}
import com.xebialabs.deployit.inspection.service.discovery.DiscoveryMessages.{DiscoveryResultsFound, RetrieveResults}

object DiscoveryResultsRetriever {
  val name = "discovery-results-fetcher"

  def props(workerManager: ActorRef, shouldNotChunk: Boolean): Props = Props(new DiscoveryResultsRetriever(workerManager, shouldNotChunk))
}

class DiscoveryResultsRetriever(workerManager: ActorRef, shouldNotChunk: Boolean) extends Actor {

  override def receive: Receive = {
    case RetrieveResults(taskId) =>
      publish(context.actorOf(Props(classOf[RetrieveResultsAggregator], taskId, shouldNotChunk)))
  }

  private def publish(aggregator: ActorRef): Unit = {
    workerManager.forward(Publish(DiscoveryResultsActor.name, aggregator))
  }
}

class RetrieveResultsAggregator(taskId: TaskId, shouldNotChunk: Boolean) extends FindOneAggregator[DiscoveryResult](RetrieveResults(taskId), shouldNotChunk) with MessageSequenceReceiver {
  override protected def create(arg: Option[DiscoveryResult]): Found[DiscoveryResult] = DiscoveryResultsFound(arg)

  override def waitForResponses(workers: Set[String], origSender: ActorRef, timeoutCancel: Cancellable, found: Boolean): Receive = receiveChunks orElse super.waitForResponses(workers, origSender, timeoutCancel, found)
}


object DiscoveryResultsActor {
  val name = "discovery-results"

  def props(): Props = Props(new DiscoveryResultsActor)
}

class DiscoveryResultsActor extends Actor {
  override def receive: Receive = {
    case RetrieveResults(taskId) =>
      sender() ! DiscoveryResultsFound(Some(DiscoveryResults.retrieve(taskId)))
  }
}

object DiscoveryMessages {

  case class RetrieveResults(taskId: TaskId)

  case class DiscoveryResultsFound(discoveryResult: Option[DiscoveryResult]) extends Found[DiscoveryResult](discoveryResult)

}

object DiscoveryWorker {
  def initialize(system: ActorSystem): Unit = {
    system.actorOf(DiscoveryResultsActor.props(), DiscoveryResultsActor.name)
  }
}
