package com.xebialabs.deployit.taskexecution.security;

import com.xebialabs.deployit.exception.NotFoundException;
import com.xebialabs.deployit.security.PermissionEnforcer;
import com.xebialabs.deployit.security.authentication.AuthenticationFailureException;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.AuthenticationProvider;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.authority.mapping.GrantedAuthoritiesMapper;
import org.springframework.security.core.authority.mapping.NullAuthoritiesMapper;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.stereotype.Component;

import java.util.Collection;

import static java.util.Collections.singletonList;

@Component
public class TaskWorkerAuthenticationProvider implements AuthenticationProvider {

    private static final Logger logger = LoggerFactory.getLogger(TaskWorkerAuthenticationProvider.class);

    private final UserDetailsService userDetailsService;

    private final GrantedAuthoritiesMapper authoritiesMapper = new NullAuthoritiesMapper();

    @Autowired
    public TaskWorkerAuthenticationProvider(final UserDetailsService userDetailsService) {
        this.userDetailsService = userDetailsService;
    }

    @Override
    public Authentication authenticate(final Authentication authentication) throws AuthenticationException {
        final TaskWorkerAuthenticationToken tokenContainer = (TaskWorkerAuthenticationToken) authentication;
        final String taskOwner = (String) tokenContainer.getCredentials();
        if (StringUtils.isNotBlank(taskOwner)) {
            logger.debug("Received Task-Owner {}", taskOwner);
            try {
                Collection<? extends GrantedAuthority> mappedAuthorities;
                if ("admin".equals(taskOwner)) {
                    mappedAuthorities = authoritiesMapper.mapAuthorities(
                            singletonList(new SimpleGrantedAuthority(PermissionEnforcer.ROLE_ADMIN))
                    );
                } else {
                    mappedAuthorities = userDetailsService.loadUserByUsername(taskOwner).getAuthorities();
                }
                return new TaskWorkerAuthenticationToken(taskOwner, mappedAuthorities);
            } catch (AuthenticationFailureException | NotFoundException | UsernameNotFoundException exc) {
                throw new BadCredentialsException("Cannot authenticate " + taskOwner, exc);
            }
        }
        return null;
    }

    @Override
    public boolean supports(final Class<?> authentication) {
        return authentication.isAssignableFrom(TaskWorkerAuthenticationToken.class);
    }

}
