package com.xebialabs.deployit.config

import ai.digital.config.AbstractEnvironmentPreparedProcessor
import ai.digital.deploy.core.common.XldServerPaths._
import com.xebialabs.deployit.env.TaskEngineActiveProfileRegistry
import com.xebialabs.deployit.{ServerConfiguration, TaskEngineLaunchOptions}
import org.springframework.boot.{ConfigurableBootstrapContext, SpringApplication}
import org.springframework.core.Ordered
import org.springframework.core.annotation.Order
import org.springframework.core.env.ConfigurableEnvironment

@Order(Ordered.LOWEST_PRECEDENCE)
class TaskEngineEnvironmentPreparedProcessor(application: SpringApplication, args: Array[String]) extends AbstractEnvironmentPreparedProcessor {

  protected override def resolveSpringCloudUri(bootstrapContext: ConfigurableBootstrapContext, environment: ConfigurableEnvironment): Option[String] =
    Option(TaskEngineLaunchOptions.getInstance)
      .flatMap(options =>
        if (ServerConfiguration.getInstance().getSpringCloudExternalConfig) {
          super.resolveSpringCloudUri(bootstrapContext, environment)
        } else {
          getSpringCloudServerUri(options)
        }
      )

  private def getSpringCloudServerUri(options: TaskEngineLaunchOptions): Some[String] = {
    val protocol = options.api.getProtocol
    val context = if (options.api.getPath != null) s"${options.api.getPath}" else ""
    val slash = if (options.api.toString.endsWith("/")) "" else "/"
    val serverUrl: String = if (options.skipPortInApi) String.format("%s://%s%s", protocol, options.api.getHost, context) else
      String.format("%s://%s:%d%s", protocol, options.api.getHost, getPort(options.api.getPort, protocol), context)
    Some(s"${serverUrl}${slash}${CENTRAL_CONFIG_PATH}")
  }

  private def getPort(port: Int, protocol: String): Int = {
    if (port == -1) {
      protocol match {
        case "https" => 4517
        case "http" => 4516
      }
    } else {
      port
    }
  }


  protected override def resolveSpringCloudConfigUsername(bootstrapContext: ConfigurableBootstrapContext, environment: ConfigurableEnvironment): Option[String] =
    if (Option(TaskEngineLaunchOptions.getInstance).isDefined &&
      TaskEngineActiveProfileRegistry.allActiveProfiles(ServerConfiguration.getInstance()).sameElements(environment.getActiveProfiles)) {
      super.resolveSpringCloudConfigUsername(bootstrapContext, environment)
    } else {
      None
    }

  protected override def resolveSpringCloudConfigPassword(bootstrapContext: ConfigurableBootstrapContext, environment: ConfigurableEnvironment): Option[String] =
    if (Option(TaskEngineLaunchOptions.getInstance).isDefined &&
      TaskEngineActiveProfileRegistry.allActiveProfiles(ServerConfiguration.getInstance()).sameElements(environment.getActiveProfiles)) {
      super.resolveSpringCloudConfigPassword(bootstrapContext, environment)
    } else {
      None
    }

  override protected def propertyResourceName: String = "default-worker-env"
}
