package com.xebialabs.deployit

import akka.actor.Extension
import com.typesafe.config.{Config, ConfigFactory}
import com.xebialabs.deployit.tasksystem.ConfigLoading
import com.xebialabs.xlplatform.config.{ConfigurationHolder, ProductConfiguration}
import com.xebialabs.xlplatform.distribution.RSAUtils
import org.bouncycastle.crypto.params.RSAKeyParameters

import java.net.URL
import scala.jdk.CollectionConverters._

object TaskEngine {
  private var instance: TaskEngine = _

  def setInstance(taskEngine: TaskEngine): Unit = {
    instance = taskEngine
  }

  def getInstance: TaskEngine = instance

  def configFromOptions(options: TaskEngineLaunchOptions): Config = {
    val map: Map[String, Any] = Map(
      "server.hostname" -> options.hostname,
      "server.port" -> options.port,
      "deploy.task.recovery-dir" -> options.workDir,
      "xl.worker.name" -> options.name,
      "deploy.server.ssl.key-store" -> options.keyStore,
      "deploy.server.ssl.key-password" -> options.keyPassword,
      "deploy.server.ssl.key-store-password" -> options.keyStorePassword,
      "deploy.server.ssl.trust-store" -> options.trustStore,
      "deploy.server.ssl.trust-store-password" -> options.trustStorePassword,
    ).filter { case (_, value) => value != null }
    ConfigFactory.parseMap(map.asJava)
  }
}

class TaskEngine(xldProductConfiguration: ProductConfiguration) {
  val config: Config = loadConfig()

  if (config.getBoolean("deploy.task.in-process-worker")) {
    throw new IllegalArgumentException("External worker cannot be started when configuration specifies internal worker ('deploy.task.in-process-worker = true'). Ensure that internal worker is disabled in master and worker configurations.")
  }

  val settings = new TaskEngineSettings(config)

  val launchOptions: TaskEngineLaunchOptions = TaskEngineLaunchOptions.getInstance
  val restApi: URL = launchOptions.api
  val socketTimeout: Integer = launchOptions.socketTimeout
  val address = s"${settings.protocol}://task-sys@${settings.hostname}:${settings.port}"
  val masters: Seq[String] = if (launchOptions.masters != null) launchOptions.masters.toSeq else Nil

  def hostnameToActorPath(hostname: String): String = s"${settings.protocol}://task-sys@$hostname/user/"

  private val taskEngineSecurity = new TaskEngineSecurity()
  val publicKey: Array[Byte] = taskEngineSecurity.publicKeyEncoded
  val privateKey: RSAKeyParameters = taskEngineSecurity.privateKey

  private def loadConfig(): Config = {
    new ConfigurationHolder(xldProductConfiguration).init()
    xldProductConfiguration.configuration
  }

  class TaskEngineSecurity() {
    private val taskEngineKeyPair = RSAUtils.generateKeyPair
    val publicKeyEncoded: Array[Byte] = taskEngineKeyPair.publicKeyEncoded
    val privateKey: RSAKeyParameters = taskEngineKeyPair.privateKey
  }

  class TaskEngineSettings(config: Config) extends Extension {
    lazy val hostname: String = config.getString("server.hostname")
    lazy val port: String = config.getString("server.port")
    lazy val protocol: String = "akka"
    lazy val name: Option[String] = if (config.hasPath("xl.worker.name") && config.getString("xl.worker.name").nonEmpty) Option(config.getString("xl.worker.name")) else None
  }
}
