package com.xebialabs.deployit.taskexecution.security;

import com.google.common.collect.Lists;
import com.xebialabs.deployit.engine.api.distribution.TaskExecutionWorkerRepository;
import com.xebialabs.deployit.engine.api.dto.Worker;
import com.xebialabs.xlplatform.distribution.RSAUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.authentication.ProviderNotFoundException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.web.authentication.AbstractAuthenticationProcessingFilter;
import scala.Option;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.Collections;


public class TaskWorkerAuthenticationFilter extends AbstractAuthenticationProcessingFilter {

    private static final String RSA_HEADER = "rsa";
    private static final String DELIMITER = ":";
    private static final int RSA_SIGNATURE_PARTS_SIZE = 6;
    private final TaskExecutionWorkerRepository taskExecutionWorkerRepository;

    public TaskWorkerAuthenticationFilter(AuthenticationManager authenticationManager, TaskExecutionWorkerRepository taskExecutionWorkerRepository) {
        super("/deployit/**");
        this.taskExecutionWorkerRepository = taskExecutionWorkerRepository;
        setAuthenticationManager(authenticationManager);
    }

    @Override
    public void doFilter(final ServletRequest request, final ServletResponse response, final FilterChain chain) throws IOException, ServletException {
        final String rsaHeader = getRSAHeader((HttpServletRequest) request);

        if (StringUtils.isEmpty(rsaHeader)) {
            chain.doFilter(request, response);
            return;
        }

        final String[] rsaSignatureParts = rsaHeader.split(DELIMITER);
        if (rsaSignatureParts.length < RSA_SIGNATURE_PARTS_SIZE) {
            chain.doFilter(request, response);
            return;
        }

        if (isInvalidRSASignature(rsaSignatureParts, (HttpServletRequest) request)) {
            chain.doFilter(request, response);
            return;
        }

        this.setAuthenticationSuccessHandler((request1, response1, authentication) -> chain.doFilter(request1, response1));

        super.doFilter(request, response, chain);
    }

    @Override
    public Authentication attemptAuthentication(final HttpServletRequest request, final HttpServletResponse response) throws AuthenticationException {
        final String rsaHeader = getRSAHeader(request);

        if (StringUtils.isEmpty(rsaHeader)) {
            return null;
        }

        final String[] rsaSignatureParts = rsaHeader.split(DELIMITER);
        if (rsaSignatureParts.length < RSA_SIGNATURE_PARTS_SIZE) {
            return null;
        }

        final String taskOwner = rsaSignatureParts[0];

        if (isInvalidRSASignature(rsaSignatureParts, request)) {
            return null;
        }

        final TaskWorkerAuthenticationToken token = new TaskWorkerAuthenticationToken(taskOwner);
        token.setDetails(authenticationDetailsSource.buildDetails(request));

        try {
            return this.getAuthenticationManager().authenticate(token);
        } catch (ProviderNotFoundException e) {
            logger.error("No authentication provider found for : " + token.getName() +
                    ". Please check <security:authentication-provider> settings in conf/deployit-security.xml");
            throw e;
        }
    }

    private boolean isInvalidRSASignature(final String[] rsaSignatureParts, final HttpServletRequest request) {
        final String workerAddress = rsaSignatureParts[1].concat(DELIMITER).concat(rsaSignatureParts[2]).concat(DELIMITER).concat(rsaSignatureParts[3]);
        final String nonce = rsaSignatureParts[4].concat(DELIMITER).concat(rsaSignatureParts[5]);
        final String rsaSignature = rsaSignatureParts[6];
        
        final String originalRequestURI = request.getQueryString() == null ? request.getRequestURI() : request.getRequestURI() + "?" + request.getQueryString();

        final String message = String.join(DELIMITER, Lists.newArrayList(originalRequestURI, request.getMethod(), request.getProtocol(), nonce));

        final Option<Worker> worker = taskExecutionWorkerRepository.getWorkerByAddress(workerAddress);
        if (worker.isDefined() &&  worker.get().publicKey() != null) {
            try {
                return !RSAUtils.verify(message, rsaSignature, worker.get().publicKey());
            } catch (Exception e) {
                logger.error("Failed to verify RSA Signature", e);
            }
        }
        return true;
    }

    private String getRSAHeader(HttpServletRequest request) {
        return Collections.list(request.getHeaderNames()).stream()
                .filter(header -> header.equalsIgnoreCase(RSA_HEADER))
                .map(request::getHeader)
                .findFirst()
                .orElse(null);
    }

}
