package com.xebialabs.deployit

import akka.actor.Extension
import com.typesafe.config.{Config, ConfigFactory}
import com.xebialabs.deployit.tasksystem.ConfigLoading
import com.xebialabs.xlplatform.config.{ConfigurationHolder, ProductConfiguration}
import com.xebialabs.xlplatform.distribution.RSAUtils
import org.bouncycastle.crypto.params.RSAKeyParameters

import java.net.URL
import scala.jdk.CollectionConverters._

object TaskEngine {
  private var instance: TaskEngine = _

  def setInstance(taskEngine: TaskEngine): Unit = {
    instance = taskEngine
  }

  def getInstance: TaskEngine = instance

  def configFromOptions(options: TaskEngineLaunchOptions): Config = {
    val map: Map[String, Any] = Map(
      "deploy.server.hostname" -> options.hostname,
      "deploy.server.port" -> options.port,
      "deploy.task.recovery-dir" -> options.workDir,
      "xl.worker.name" -> options.name
    ).filter { case (_, value) => value != null }
    ConfigFactory.parseMap(map.asJava)
  }
}

class TaskEngine(xldProductConfiguration: ProductConfiguration) {
  val launchOptions: TaskEngineLaunchOptions = TaskEngineLaunchOptions.getInstance
  val config: Config = loadConfig(launchOptions)

  if (config.getBoolean("deploy.task.in-process-worker")) {
    throw new IllegalArgumentException("External worker cannot be started when configuration specifies internal worker ('deploy.task.in-process-worker = true'). Ensure that internal worker is disabled in master and worker configurations.")
  }

  val settings = new TaskEngineSettings(config)

  val restApi: URL = launchOptions.api
  val socketTimeout: Integer = launchOptions.socketTimeout
  val address = s"${settings.protocol}://task-sys@${settings.hostname}:${settings.port}"
  val masters: Seq[String] = if (launchOptions.masters != null) launchOptions.masters.toSeq else Nil

  def hostnameToActorPath(hostname: String): String = s"${settings.protocol}://task-sys@$hostname/user/"

  private val taskEngineSecurity = new TaskEngineSecurity()
  val publicKey: Array[Byte] = taskEngineSecurity.publicKeyEncoded
  val privateKey: RSAKeyParameters = taskEngineSecurity.privateKey

  private def loadConfig(options: TaskEngineLaunchOptions): Config = {
    val config = TaskEngine.configFromOptions(options).
      withFallback(Option(options.configFile).map(ConfigFactory.parseFile).getOrElse(ConfigFactory.parseResources("xl-worker.conf"))).
      withFallback(xldProductConfiguration.configuration)
    new ConfigurationHolder(xldProductConfiguration).init()
    ConfigLoading.loadConfig(config)
  }

  class TaskEngineSecurity() {
    private val taskEngineKeyPair = RSAUtils.generateKeyPair
    val publicKeyEncoded: Array[Byte] = taskEngineKeyPair.publicKeyEncoded
    val privateKey: RSAKeyParameters = taskEngineKeyPair.privateKey
  }

  class TaskEngineSettings(config: Config) extends Extension {
    lazy val hostname: String = config.getString("deploy.server.hostname")
    lazy val port: String = config.getString("deploy.server.port")
    lazy val protocol: String = "akka"
    lazy val name: Option[String] = if (config.hasPath("xl.worker.name") && config.getString("xl.worker.name").nonEmpty) Option(config.getString("xl.worker.name")) else None
  }
}
