package com.xebialabs.deployit.task;

import ai.digital.deploy.tasker.common.TaskType;
import com.google.common.annotations.VisibleForTesting;
import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.engine.api.execution.FetchMode;
import com.xebialabs.deployit.engine.api.execution.TaskExecutionState;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.util.Tuple;
import org.joda.time.DateTime;

import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;

import static com.google.common.collect.Sets.newHashSet;
import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static com.xebialabs.deployit.core.util.CiUtils.getName;

public class ArchivedTaskSearchParameters {

    public enum DateRange {
        AFTER, BEFORE, BETWEEN, NONE
    }

    public enum Status {
        COMPLETED, CANCELLED, COMPLETED_AFTER_RETRY, ANY
    }

    protected DateTime startDate;
    protected DateTime endDate;
    protected String executedBy;
    protected Set<String> users = newHashSet();
    protected Set<Application> applications = newHashSet();
    protected Set<String> environments = newHashSet();
    protected String taskUuid;
    protected DateRange dateRangeSearch = ArchivedTaskSearchParameters.DateRange.NONE;
    protected Status status = ArchivedTaskSearchParameters.Status.ANY;
    protected Set<TaskExecutionState> taskExecutionStates = EnumSet.noneOf(TaskExecutionState.class);
    protected Set<TaskType> taskTypes = EnumSet.noneOf(TaskType.class);
    protected List<Tuple<String, Boolean>> orderBy = new ArrayList<>();
    protected int page;
    protected int resultsPerPage;
    protected boolean onlySuccessful;
    protected FetchMode fetchMode;
    protected Set<String> targets;
    protected String taskName;
    protected String workerName;
    protected List<String> references;
    protected boolean filterByArchivedDate;

    public ArchivedTaskSearchParameters() {
        this.filterByArchivedDate = false;
    }

    public ArchivedTaskSearchParameters(ArchivedTaskSearchParameters p) {
        this.startDate = p.startDate;
        this.endDate = p.endDate;
        this.executedBy = p.executedBy;
        this.applications = newHashSet(p.applications);
        this.environments = newHashSet(p.environments);
        this.dateRangeSearch = p.dateRangeSearch;
        this.status = p.status;
        this.taskExecutionStates = p.taskExecutionStates;
        this.taskUuid = p.taskUuid;
        this.taskTypes = p.taskTypes;
        this.orderBy.addAll(p.orderBy);
        this.page = p.page;
        this.resultsPerPage = p.resultsPerPage;
        this.users = p.users;
        this.onlySuccessful = p.onlySuccessful;
        this.fetchMode = p.fetchMode;
        this.targets = p.targets;
        this.taskName = p.taskName;
        this.workerName = p.workerName;
        this.references = p.references;
        this.filterByArchivedDate = p.filterByArchivedDate;
    }

    public ArchivedTaskSearchParameters orderBy(String property, boolean ascending) {
        this.orderBy.add(new Tuple<>(property, ascending));
        return this;
    }

    public ArchivedTaskSearchParameters executedByUser(String user) {
        this.executedBy = user;
        return this;
    }

    public ArchivedTaskSearchParameters createdBetween(DateTime startDate, DateTime endDate) {
        checkNotNull(startDate, "Start date cannot be null");
        checkNotNull(endDate, "End date cannot be null");
        dateRangeSearch = ArchivedTaskSearchParameters.DateRange.BETWEEN;
        if (startDate.isAfter(endDate)) {
            this.startDate = endDate;
            this.endDate = startDate;
        } else {
            this.startDate = startDate;
            this.endDate = endDate;
        }
        return this;
    }

    public ArchivedTaskSearchParameters createdOnOrBefore(DateTime endDate) {
        checkNotNull(endDate, "End date cannot be null");
        dateRangeSearch = ArchivedTaskSearchParameters.DateRange.BEFORE;
        this.endDate = endDate;
        return this;
    }

    public ArchivedTaskSearchParameters createdOnOrAfter(DateTime startDate) {
        checkNotNull(startDate, "Start date cannot be null");
        dateRangeSearch = ArchivedTaskSearchParameters.DateRange.AFTER;
        this.startDate = startDate;
        return this;
    }

    /**
     * Sets the datetime search constraints to BEFORE, AFTER, BETWEEN or not at all, depending on
     * whether <tt>startDate</tt> or <tt>endDate</tt> (or both) are null
     *
     * @param startDate the datetime range start; can be null
     * @param endDate   the datetime range end; can be null
     * @return {@code this}
     */
    public ArchivedTaskSearchParameters inDateTimeRange(DateTime startDate, DateTime endDate) {
        if (startDate == null && endDate == null) {
            return this;
        } else if (startDate == null) {
            return this.createdOnOrBefore(endDate);
        } else if (endDate == null) {
            return this.createdOnOrAfter(startDate);
        } else {
            return this.createdBetween(startDate, endDate);
        }
    }

    public ArchivedTaskSearchParameters forApplicationAndVersion(String application, String version) {
        applications.add(new Application(application, version));
        return this;
    }

    public ArchivedTaskSearchParameters forApplication(String application) {
        return forApplicationAndVersion(application, null);
    }

    public ArchivedTaskSearchParameters thatAreOfType(Set<TaskType> types) {
        this.taskTypes = types;
        return this;
    }

    public ArchivedTaskSearchParameters forEnvironment(String environment) {
        environments.add(environment);
        return this;
    }

    public ArchivedTaskSearchParameters forFilterType(FilterType filterType, List<ConfigurationItemId> cis) {
        for (ConfigurationItemId appOrEnvName : cis) {
            forFilterType(filterType, appOrEnvName);
        }

        return this;
    }

    public ArchivedTaskSearchParameters forFilterType(FilterType filterType, ConfigurationItemId appOrEnvName) {
        switch (filterType) {
            case APPLICATION:
                Checks.checkTrue(environments.isEmpty(), "Can't query for both environments and applications without FilterType.BOTH.");
                forApplication(getName(appOrEnvName.getId()));
                break;
            case ENVIRONMENT:
                Checks.checkTrue(applications.isEmpty(), "Can't query for both environments and applications without FilterType.BOTH.");
                forEnvironment(appOrEnvName.getId());
                break;
            case BOTH:
                Type application = Type.valueOf("udm.Application");
                if (appOrEnvName.getType().equals(application) || appOrEnvName.getType().isSubTypeOf(application)) {
                    forApplication(getName(appOrEnvName.getId()));
                } else {
                    forEnvironment(appOrEnvName.getId());
                }
                break;
            default:
                break;
        }
        return this;
    }

    public ArchivedTaskSearchParameters withTaskStates(List<String> states) {
        for (String state : states) {
            this.taskExecutionStates.add(TaskExecutionState.valueOf(state.toUpperCase()));
        }
        return this;
    }

    public ArchivedTaskSearchParameters thatWasCancelled() {
        status = Status.CANCELLED;
        return this;
    }

    public ArchivedTaskSearchParameters thatCompleted() {
        status = Status.COMPLETED;
        return this;
    }

    public ArchivedTaskSearchParameters thatCompletedAfterRetry() {
        status = Status.COMPLETED_AFTER_RETRY;
        return this;
    }

    public ArchivedTaskSearchParameters withUniqueId(String uuid) {
        this.taskUuid = uuid;
        return this;
    }

    public ArchivedTaskSearchParameters showPage(int page, int resultsPerPage) {
        this.page = page;
        this.resultsPerPage = resultsPerPage;
        return this;
    }

    public ArchivedTaskSearchParameters forTaskTypes(Set<TaskType> taskTypes) {
        this.taskTypes = taskTypes;
        return this;
    }

    public ArchivedTaskSearchParameters forUsers(Set<String> users) {
        this.users = users;
        return this;
    }

    public ArchivedTaskSearchParameters forSuccessfulOnly(boolean onlySuccessful) {
        this.onlySuccessful = onlySuccessful;
        return this;
    }

    public ArchivedTaskSearchParameters withFetchMode(FetchMode fetchMode) {
        this.fetchMode = fetchMode;
        return this;
    }

    public ArchivedTaskSearchParameters withTargets(Set<String> targets) {
        this.targets = targets;
        return this;
    }

    public ArchivedTaskSearchParameters withTaskName(String taskName) {
        this.taskName = taskName;
        return this;
    }

    public ArchivedTaskSearchParameters withWorkerName(String workerName) {
        this.workerName = workerName;
        return this;
    }

    public ArchivedTaskSearchParameters applySecurityParameters(List<String> references) {
        this.references = references;
        return this;
    }

    public ArchivedTaskSearchParameters setFilterByArchivedDate(boolean filterByArchivedDate) {
        this.filterByArchivedDate = filterByArchivedDate;
        return this;
    }


    public static class Application {
        public final String name;
        public final String version;

        Application(String name, String version) {
            this.name = name;
            this.version = version;
        }
    }

    public FetchMode getFetchMode() {
        return fetchMode;
    }

    @VisibleForTesting
    public Set<TaskType> getTaskTypes() {
        return taskTypes;
    }

    @VisibleForTesting
    public DateTime getStartDate() {
        return startDate;
    }

    @VisibleForTesting
    public DateTime getEndDate() {
        return endDate;
    }

    @VisibleForTesting
    public List<Tuple<String, Boolean>> getOrderBy() {
        return orderBy;
    }

    @VisibleForTesting
    public int getPage() {
        return page;
    }

    @VisibleForTesting
    public int getResultsPerPage() {
        return resultsPerPage;
    }

    @VisibleForTesting
    public String getTaskUuid() {
        return taskUuid;
    }

    @VisibleForTesting
    public boolean isOnlySuccessful() {
        return onlySuccessful;
    }

    @VisibleForTesting
    public Set<String> getUsers() {
        return users;
    }

    @VisibleForTesting
    public Set<Application> getApplications() {
        return applications;
    }

    @VisibleForTesting
    public Set<String> getEnvironments() {
        return environments;
    }

    @VisibleForTesting
    public Set<TaskExecutionState> getTaskExecutionStates() {
        return taskExecutionStates;
    }

    @VisibleForTesting
    public Set<String> getTargets() {
        return targets;
    }

    @VisibleForTesting
    public String getTaskName() {
        return taskName;
    }

    public boolean hasSecurityParameters() {
        return this.references != null;
    }

    public boolean isFilterByArchivedDate() { return this.filterByArchivedDate; }
}
