package com.xebialabs.deployit.task.archive

import java.lang
import java.lang.Boolean.FALSE

import ai.digital.deploy.tasker.common.TaskMetadata._
import com.xebialabs.deployit.engine.api.execution.TaskWithBlock
import org.joda.time.DateTime

import scala.jdk.CollectionConverters._

class DeploymentTaskArchiveSerializer(override val task: TaskWithBlock) extends DefaultTaskArchiveSerializer(task) {
  override def getTaskKind: String = DeploymentTaskArchiveSerializer.kind

  private val mainApplication = task.getMetadata.get(APPLICATION)

  def getNullableIntProperty(name: String): Option[Int] = {
    val securedCi = task.getMetadata.get(name)
    if (String.valueOf(securedCi).equals("null")) None else Option(securedCi.toInt)
  }

  override def serialize(workerName: Option[String]): TaskMessage = {
    DeploymentTaskMessage(
      task.getId,
      archiveDetails(workerName),
      mainApplication,
      getPackages.iterator.mkString(", "),
      task.getMetadata.get(ENVIRONMENT_ID),
      task.getMetadata.get(ENVIRONMENT_INTERNAL_ID).toInt,
      task.getMetadata.get(ENVIRONMENT_REFERENCE_ID),
      getNullableIntProperty(ENVIRONMENT_SECURED_CI),
      task.getMetadata.get(ENVIRONMENT_DIRECTORY_REFERENCE),
      task.getMetadata.get(TASK_TYPE),
      task.getOwner,
      task.getState.toString,
      task.getFailureCount,
      task.getStartDate,
      task.getCompletionDate,
      task.getCompletionDate.getMillis - task.getStartDate.getMillis,
      FALSE,
      applicationMessages(task)
    )
  }

  private def getPackages: IterableOnce[String] =
    applicationVersion(mainApplication, task.getMetadata.get(VERSION)) +:
      task.getPackageDependencies.asScala.reverse.map(dep => applicationVersion(dep.getApplication, dep.getVersion))

  private def applicationVersion(application: String, version: String) = s"$application/$version"

  private def applicationMessages(task: TaskWithBlock): List[ApplicationMessage] =
    (mainApplication +: task.getPackageDependencies.asScala.map(_.getApplication)).distinct.map { application =>
      // bug detected ENG-2617
      ApplicationMessage(application,
        task.getMetadata.get(APPLICATION_INTERNAL_ID).toInt,
        task.getMetadata.get(APPLICATION_REFERENCE_ID),
        getNullableIntProperty(APPLICATION_SECURED_CI),
        task.getMetadata.get(APPLICATION_DIRECTORY_REFERENCE)
      )
    }.toList
}

object DeploymentTaskArchiveSerializer {
  val kind = "DEPLOYMENT"
}

case class DeploymentTaskMessage(override val taskId: String,
                                 override val archiveDetails: ArchiveDetails,
                                 mainApplication: String,
                                 packages: String,
                                 environmentId: String,
                                 environmentInternalId: Int,
                                 environmentReferenceId: String,
                                 environmentSecuredId: Option[Int],
                                 environmentDirectoryReference: String,
                                 taskType: String,
                                 owner: String,
                                 state: String,
                                 failureCount: Integer,
                                 override val startDate: DateTime,
                                 endDate: DateTime,
                                 duration: Long,
                                 isRollback: lang.Boolean,
                                 applications: List[ApplicationMessage]) extends TaskMessage

case class ApplicationMessage(applicationName: String,
                              applicationInternalId: Int,
                              applicationReferenceId: String,
                              applicationSecuredId: Option[Int],
                              applicationDirectoryReference: String)
