package com.xebialabs.deployit.task.archive

import java.lang
import java.lang.Boolean.FALSE

import com.xebialabs.deployit.engine.api.execution.TaskWithBlock
import com.xebialabs.deployit.task.TaskMetadata
import org.joda.time.DateTime

import scala.jdk.CollectionConverters._

class DeploymentTaskArchiveSerializer(override val task: TaskWithBlock) extends DefaultTaskArchiveSerializer(task) {
  override def getTaskKind: String = DeploymentTaskArchiveSerializer.kind

  private val mainApplication = task.getMetadata.get(TaskMetadata.APPLICATION)

  def getNullableIntProperty(name: String): Option[Int] = {
    val securedCi = task.getMetadata.get(name)
    if (String.valueOf(securedCi).equals("null")) None else Option(securedCi.toInt)
  }

  override def serialize(workerName: Option[String]): TaskMessage = {
    DeploymentTaskMessage(
      task.getId,
      archiveDetails(workerName),
      mainApplication,
      getPackages.iterator.mkString(", "),
      task.getMetadata.get(TaskMetadata.ENVIRONMENT_ID),
      task.getMetadata.get(TaskMetadata.ENVIRONMENT_INTERNAL_ID).toInt,
      getNullableIntProperty(TaskMetadata.ENVIRONMENT_SECURED_CI),
      task.getMetadata.get(TaskMetadata.ENVIRONMENT_DIRECTORY_REFERENCE),
      task.getMetadata.get(TaskMetadata.TASK_TYPE),
      task.getOwner,
      task.getState.toString,
      task.getFailureCount,
      task.getStartDate,
      task.getCompletionDate,
      task.getCompletionDate.getMillis - task.getStartDate.getMillis,
      FALSE,
      applicationMessages(task)
    )
  }

  private def getPackages: IterableOnce[String] =
    applicationVersion(mainApplication, task.getMetadata.get(TaskMetadata.VERSION)) +:
      task.getPackageDependencies.asScala.reverse.map(dep => applicationVersion(dep.getApplication, dep.getVersion))

  private def applicationVersion(application: String, version: String) = s"$application/$version"

  private def applicationMessages(task: TaskWithBlock): List[ApplicationMessage] =
    (mainApplication +: task.getPackageDependencies.asScala.map(_.getApplication)).distinct.map { application =>
      // bug detected ENG-2617
      ApplicationMessage(application,
        task.getMetadata.get(TaskMetadata.APPLICATION_INTERNAL_ID).toInt,
        getNullableIntProperty(TaskMetadata.APPLICATION_SECURED_CI),
        task.getMetadata.get(TaskMetadata.APPLICATION_DIRECTORY_REFERENCE)
      )
    }.toList
}

object DeploymentTaskArchiveSerializer {
  val kind = "DEPLOYMENT"
}

case class DeploymentTaskMessage(override val taskId: String,
                                 override val archiveDetails: ArchiveDetails,
                                 mainApplication: String,
                                 packages: String,
                                 environmentId: String,
                                 environmentInternalId: Int,
                                 environmentSecuredId: Option[Int],
                                 environmentDirectoryReference: String,
                                 taskType: String,
                                 owner: String,
                                 state: String,
                                 failureCount: Integer,
                                 override val startDate: DateTime,
                                 endDate: DateTime,
                                 duration: Long,
                                 isRollback: lang.Boolean,
                                 applications: List[ApplicationMessage]) extends TaskMessage

case class ApplicationMessage(applicationName: String,
                              applicationInternalId: Int,
                              applicationSecuredId: Option[Int],
                              applicationDirectoryReference: String)
