package com.xebialabs.deployit.task.archive

import java.util

import com.xebialabs.deployit.engine.api.execution.TaskWithBlock
import com.xebialabs.deployit.task.TaskMetadata._
import com.xebialabs.deployit.task.TaskType
import com.xebialabs.deployit.task.TaskType._
import org.apache.commons.lang.{StringUtils, Validate}
import org.joda.time.DateTime

object TaskArchiveSerlializer {
  def apply(task: TaskWithBlock): TaskSerlializer = task match {
    case DeploymentTask(deploymentSerializer) => deploymentSerializer
    case ControlTask(controlTaskSerializer) => controlTaskSerializer
    case _ => new DefaultTaskArchiveSerializer(task)
  }
}

object ControlTask {

  val CONTROL_TASK_TYPES: util.EnumSet[TaskType] = util.EnumSet.of(CONTROL)

  def unapply(task: TaskWithBlock): Option[TaskSerlializer] = {
    if (CONTROL_TASK_TYPES.contains(TaskType.valueOf(getMetadata(task, TASK_TYPE))) && !metadataContainsKey(task, CLOUD_OPERATION)) {
      Validate.isTrue(StringUtils.isNotEmpty(getMetadata(task, TASK_NAME)), "task name must be set for archiving of control tasks")
      Validate.isTrue(StringUtils.isNotEmpty(getMetadata(task, CONTROL_TASK_TARGET_CI)), "target CI id must be set for archiving of control tasks")
      Some(new ControlTaskArchiveSerializer(task))
    } else {
      None
    }
  }
}

object DeploymentTask {

  val DEPLOYMENT_TASK_TYPES: util.EnumSet[TaskType] = util.EnumSet.of(ROLLBACK, INITIAL, UPGRADE, UNDEPLOY)

  def unapply(task: TaskWithBlock): Option[TaskSerlializer] = {
    val taskType = TaskType.valueOf(getMetadata(task, TASK_TYPE))
    if (DEPLOYMENT_TASK_TYPES.contains(taskType)) {
      Validate.isTrue(StringUtils.isNotEmpty(getMetadata(task, APPLICATION)), "applicationName in deployment task must be set")
      Validate.isTrue(StringUtils.isNotEmpty(getMetadata(task, VERSION)), "applicationVersion in deployment task must be set")
      Validate.isTrue(StringUtils.isNotEmpty(getMetadata(task, ENVIRONMENT)), "environment in deployment task must be set")
      Validate.isTrue(StringUtils.isNotEmpty(getMetadata(task, ENVIRONMENT_ID)), "environment_id in deployment task must be set")

      Some(new DeploymentTaskArchiveSerializer(task))
    } else {
      None
    }
  }
}

trait TaskSerlializer {
  def serialize(workerName: Option[String] = None): TaskMessage
}

case class ArchiveDetails(taskKind: String,
                          workerName: Option[String],
                          taskDetails: String)

trait TaskMessage {
  val archiveDetails: ArchiveDetails
  val taskId: String
  val startDate: DateTime
}

case class SecureCiMessage(securedCi: Integer,
                           primaryKey: Option[Integer],
                           environmentIds: Option[List[Integer]],
                           applicationIds: Option[List[Integer]],
                           otherCiIds: Option[List[Integer]])

case class BatchSecureCiMessage(environmentBatchMessages: List[BatchEnvironmentMessage],
                                applicationBatchMessages: List[BatchApplicationMessage],
                                controlTaskBatchMessages: List[BatchControlTaskMessage])

case class BatchEnvironmentMessage(override val securedCi: Integer,
                                   override val primaryKey: Integer) extends BatchMessage(securedCi, primaryKey)

case class BatchApplicationMessage(override val securedCi: Integer,
                                   override val primaryKey: Integer) extends BatchMessage(securedCi, primaryKey)

case class BatchControlTaskMessage(override val securedCi: Integer,
                                   override val primaryKey: Integer) extends BatchMessage(securedCi, primaryKey)

class BatchMessage(val securedCi: Integer,
                   val primaryKey: Integer)
