package com.xebialabs.deployit.task.archive.queue

import com.xebialabs.deployit.engine.api.distribution.TaskExecutionWorkerRepository
import com.xebialabs.deployit.engine.api.execution.TaskWithBlock
import com.xebialabs.deployit.task.archive._
import com.xebialabs.deployit.task.archive.compression.CompressionUtils
import com.xebialabs.xldeploy.jms.adapter.ArchiveQueueNameResolver
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jms.core.{JmsTemplate, MessagePostProcessor}
import org.springframework.stereotype.Service

import jakarta.jms.Message

@Service
@Autowired
class AsyncTaskArchiveProducer(@Qualifier("mainJmsTemplate") jmsTemplate: JmsTemplate,
                               archiveQueueNameResolver: ArchiveQueueNameResolver,
                               workerRepository: TaskExecutionWorkerRepository) extends TaskArchiveStore with Logging {
  override def archive(task: TaskWithBlock): Unit = {
    val taskMessage = TaskArchiveSerlializer(task).serialize(workerRepository.getWorker(task.getWorkerId).map(_.name))
    logger.debug(s"Sending Archive Message for taskId [${taskMessage.taskId}]")
    val compressedMessage = CompressionUtils.compressTaskMessage(taskMessage)
    jmsTemplate.convertAndSend(archiveQueueNameResolver.getQueueName, compressedMessage, postProcessMessage(taskMessage.taskId))
  }

  override def updateSecureCi(securedCi: Integer,
                              environmentIds: Option[List[Integer]],
                              applicationIds: Option[List[Integer]],
                              otherCiIds: Option[List[Integer]]): Unit =
    updateSecureCi(
      securedCi,
      None,
      environmentIds,
      applicationIds,
      otherCiIds)

  override def updateSecureCi(pk: Integer, securedCi: Integer): Unit =
    updateSecureCi(securedCi, Option(pk), None, None, None)


  private def updateSecureCi(securedCi: Integer,
                             primaryKey: Option[Integer],
                             environmentIds: Option[List[Integer]],
                             applicationIds: Option[List[Integer]],
                             otherCiIds: Option[List[Integer]]): Unit = {
    val secureCiMessage = SecureCiMessage(securedCi, primaryKey, environmentIds, applicationIds, otherCiIds)
    logger.debug(s"Sending Archive Secure CI for securedCi: [$securedCi]")
    jmsTemplate.convertAndSend(archiveQueueNameResolver.getQueueName, secureCiMessage, postProcessMessage(primaryKey.isEmpty))
  }

  override def batchUpdateSecureCi(batchMessages: Iterable[BatchMessage]): Unit = {
    if (!batchMessages.isEmpty) {
      val messageMap = batchMessages groupBy {
        case _: BatchEnvironmentMessage => BatchEnvironmentMessage.getClass.getName
        case _: BatchApplicationMessage => BatchApplicationMessage.getClass.getName
        case _: BatchControlTaskMessage => BatchControlTaskMessage.getClass.getName
      }

      jmsTemplate.convertAndSend(
        archiveQueueNameResolver.getQueueName,
        BatchSecureCiMessage(
          messageMap.getOrElse(BatchEnvironmentMessage.getClass.getName, List.empty).map(_.asInstanceOf[BatchEnvironmentMessage]).toList,
          messageMap.getOrElse(BatchApplicationMessage.getClass.getName, List.empty).map(_.asInstanceOf[BatchApplicationMessage]).toList,
          messageMap.getOrElse(BatchControlTaskMessage.getClass.getName, List.empty).map(_.asInstanceOf[BatchControlTaskMessage]).toList
        ))
    }
  }


  override def updateSecuredDirectoryReference(securedDirectoryRef: String,
                                               environmentIds: Option[List[Integer]],
                                               applicationIds: Option[List[Integer]],
                                               otherCiIds: Option[List[Integer]]): Unit = {
    updateSecuredDirectoryReference(securedDirectoryRef, None, environmentIds, applicationIds, otherCiIds)
  }


  override def updateSecuredDirectoryReference(pk: Integer, securedDirectoryRef: String): Unit =
    updateSecuredDirectoryReference(securedDirectoryRef, Option(pk), None, None, None)

  private def updateSecuredDirectoryReference(securedDirectoryRef: String,
                             primaryKey: Option[Integer],
                             environmentIds: Option[List[Integer]],
                             applicationIds: Option[List[Integer]],
                             otherCiIds: Option[List[Integer]]): Unit = {
    val message = SecureDirectoryReferenceMessage(securedDirectoryRef, primaryKey, environmentIds, applicationIds, otherCiIds)
    logger.debug(s"Sending Archive Secure Directory Reference: [$securedDirectoryRef]")
    jmsTemplate.convertAndSend(archiveQueueNameResolver.getQueueName, message, postProcessMessage(primaryKey.isEmpty))
  }


  private def postProcessMessage(taskId: String): MessagePostProcessor = (message: Message) => {
    message.setStringProperty(ArchiveTaskQueueing.ArchivedTaskIdProperty, taskId)
    message
  }


  private def postProcessMessage(isBatch: Boolean): MessagePostProcessor = (message: Message) => {
    message.setStringProperty(ArchiveTaskQueueing.IsBatchProperty, isBatch.toString)
    message
  }
}
