package com.xebialabs.deployit.engine.tasker.log

import com.sksamuel.elastic4s.ElasticClient
import com.sksamuel.elastic4s.ElasticDsl._
import com.xebialabs.deployit.engine.tasker.{BlockPath, TaskId}
import grizzled.slf4j.Logging

import java.time.Instant
import scala.concurrent.duration._
import scala.util.{Failure, Success}

package object elastic extends Logging {

  final val TASK_ID = "taskId"
  final val STEP_PATH = "stepPath"
  final val FAILURE_COUNT = "failureCount"
  final val MESSAGE = "message"
  final val TIMESTAMP = "@timestamp"

  implicit val timeout: FiniteDuration = 10.seconds

  import scala.concurrent.ExecutionContext.Implicits.global

  private[log] def toLogs(sources: List[Map[String, AnyRef]]): List[String] =
    sources
      .groupBy(_(FAILURE_COUNT).asInstanceOf[Int])
      .view
      .mapValues(_.map(_(MESSAGE).asInstanceOf[String]).mkString)
      .toList
      .sortBy { case (count, _) => count }
      .map { case (_, message) => message }

  private[log] def doAppend(client: ElasticClient, logIndex: String, taskId: TaskId, stepPath: BlockPath, timestamp: Instant, failureCount: Int, message: String): Unit = {
    client.execute {
      indexInto(logIndex).fields(
        TIMESTAMP -> timestamp,
        TASK_ID -> taskId,
        STEP_PATH -> stepPath.toBlockId,
        FAILURE_COUNT -> failureCount,
        MESSAGE -> message
      )
    }.onComplete {
      case Success(response) => logger.debug(s"Successfully stored data in ES: ${response.body.getOrElse("[no response]")}")
      case Failure(e) => error("Could not write logs to external storage, logs will not be persisted", e)
    }
  }
}
