package com.xebialabs.deployit.core.rest.api;

import com.xebialabs.deployit.core.rest.api.support.TaskPermissionFilter;
import com.xebialabs.deployit.engine.api.TaskService;
import com.xebialabs.deployit.engine.api.execution.*;
import com.xebialabs.deployit.engine.tasker.Task;
import com.xebialabs.deployit.security.PermissionDeniedException;
import com.xebialabs.deployit.security.Permissions;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.security.permission.DeployitPermissions.TASK_SKIPSTEP;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN;

@Service
public class TaskResource extends AbstractTaskResource implements TaskService {

    @Override
    public TaskState getTask(final String taskId) {
        return new TaskStateView(viewAbleTask(pendingOrLiveOrArchivedTask(taskId, false)));
    }

    @Override
    public TaskState assign(final String taskId, final String owner) {
        return new TaskStateView(doAssign(taskId, owner));
    }

    @Override
    public TaskState takeover(String taskId, String owner) {
        return new TaskStateView(doTakeover(taskId, owner));
    }

    @Override
    public TaskWithSteps skip(String taskId, List<Integer> stepIds) {
        return new TaskWithStepsView(skip(taskId, stepIds, true));
    }

    @Override
    public TaskWithSteps unskip(String taskId, List<Integer> stepIds) {
        return new TaskWithStepsView(skip(taskId, stepIds, false));
    }

    @Override
    public TaskWithSteps addPause(final String taskId, final int stepNr) {
        pause(taskId, stepNr);
        return new TaskWithStepsView(engine.get().retrieve(taskId));
    }

    /**
     * Lists all unfinished tasks for the current user.
     * <p/>
     * NOTE: this method is invoked from the GUI when restarting the UI after a
     * crash. Therefore, it should only return the tasks for the currently
     * logged in user to prevent logging in as an administrator and seeing all
     * tasks in the system open in your GUI.
     * <p/>
     * If you do need the latter functionality, see getAllUnfinishedTasks().
     */
    @Override
    public List<TaskState> getMyCurrentTasks() {
        final String principal = Permissions.getAuthenticatedUserName();
        return getAllTasks().filter(task -> task.getOwner().equals(principal)).collect(Collectors.toList());
    }

    private Stream<TaskStateView> getAllTasks() {
        return engine.get().getAllIncompleteTasks(FetchMode.FULL).stream().map(TaskStateView::new);
    }

    /**
     * Lists all unfinished tasks in the system.
     */
    @Override
    public List<TaskState> getAllCurrentTasks() {
        if (hasPermission(ADMIN)) {
            logger.debug("getAllCurrentTasks returning full list as ADMIN");
            return getAllTasks().collect(Collectors.toList());
        } else {
            return getAllTasks().filter(new TaskPermissionFilter()).collect(Collectors.toList());
        }
    }

    @Override
    public TaskWithSteps getSteps(String taskId) {
        return new TaskWithStepsView(viewAbleTask(pendingOrLiveOrArchivedTask(taskId, true)), stepLogRetriever);
    }

    @Override
    public Stream<TaskState> query(LocalDate begin, LocalDate end) {
        return search(toSearchParameters(begin, end, false), false).map(TaskStateView::new);
    }

    @Override
    public Stream<TaskWithSteps> export(LocalDate begin, LocalDate end) {
        return search(toSearchParameters(begin, end, false), true).map(TaskWithStepsView::new);
    }

    protected Task skip(String taskId, List<Integer> stepIds, boolean skip) {

        final SerializableTask task = getSerializableTask(taskId);

        checkArgument(task != null, "Could not find active task %s", taskId);
        checkPermission(TASK_SKIPSTEP, task);

        if (skip) {
            engine.get().skipSteps(taskId, stepIds);
        } else {
            engine.get().unskipSteps(taskId, stepIds);
        }
        return engine.get().retrieve(taskId);
    }

    protected void pause(final String taskId, final int stepNr) {
        if (isNotCalledByOwner(taskId) && !hasPermission(ADMIN)) {
            throw PermissionDeniedException.withMessage("Only owner or ADMIN can add pause to the task.");
        }
        engine.get().addPauseStep(taskId, stepNr);
    }

    public StepState getStep(String taskId, int stepNr, DateTime ifModifiedSince) {
        StepState step = new TaskWithStepsView(viewAbleTask(pendingOrLiveOrArchivedTask(taskId, false)), stepLogRetriever).getStep(stepNr);

        if (hasBeenModifiedSince(step, ifModifiedSince)) {
            return step;
        } else {
            throw new NotModifiedException();
        }
    }
}
