package com.xebialabs.deployit.core.rest.api.support;

import ai.digital.deploy.core.common.security.permission.DeployitPermissions;
import com.xebialabs.deployit.engine.api.execution.SerializableTask;
import com.xebialabs.deployit.security.permission.Permission;
import com.xebialabs.deployit.task.TaskType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Arrays;
import java.util.function.Predicate;

import static com.xebialabs.deployit.task.TaskMetadata.*;

public class TaskPermissionFilter implements Predicate<SerializableTask> {
    private static final Logger logger = LoggerFactory.getLogger(TaskPermissionFilter.class);

    @Override
    public boolean test(SerializableTask taskState) {
        String envId = getMetadata(taskState, ENVIRONMENT_ID);
        TaskType taskType = getTaskType(taskState);
        String taskId = taskState.getId();
        if (envId == null || taskType == null) {
            logger.debug("including non deployment task {}", taskId);
            return true;
        }
        switch (taskType) {
        case INITIAL:
            return checkPermissionAndLog(taskId, envId, "initial deploy", DeployitPermissions.DEPLOY_INITIAL());
        case UPGRADE:
            return checkPermissionAndLog(taskId, envId, "upgrade", DeployitPermissions.DEPLOY_UPGRADE());
        case INSPECTION:
            return checkPermissionAndLog(taskId, "discovery", DeployitPermissions.DISCOVERY());
        case CONTROL:
            return checkPermissionAndLog(taskId, envId, "controltask", DeployitPermissions.CONTROLTASK_EXECUTE());
        case UNDEPLOY:
            return checkPermissionAndLog(taskId, envId, "undeploy", DeployitPermissions.UNDEPLOY());
        case ROLLBACK:
            return checkPermissionAndLog(taskId, envId, "undeploy", DeployitPermissions.DEPLOY_UPGRADE(), DeployitPermissions.DEPLOY_INITIAL());
        }
        return false;
    }

    private boolean checkPermissionAndLog(String taskId, String envId, String right, Permission permission) {
        if (hasPermission(permission, envId)) {
            logger.debug("including task {} user has {} rights on {}", taskId, right, envId);
            return true;
        }
        return false;
    }
    private boolean checkPermissionAndLog(String taskId, String envId, String right, Permission... permissions) {
        if (Arrays.stream(permissions).anyMatch(p -> hasPermission(p, envId))) {
            logger.debug("including task {} user has {} rights on {}", taskId, right, envId);
            return true;
        }
        return false;
    }

    private boolean checkPermissionAndLog(String taskId, String right, Permission permission) {
        if (hasPermission(permission)) {
            logger.debug("including task {} user has {} rights", taskId, right);
            return true;
        }
        return false;
    }

    private TaskType getTaskType(SerializableTask taskState) {
        try {
            String taskType = getMetadata(taskState, TASK_TYPE);
            if (taskType != null) {
                return TaskType.valueOf(taskType);
            } else {
                return null;
            }
        } catch (IllegalArgumentException e) {
            return null;
        }
    }

    protected static boolean hasPermission(Permission permission) {
        return hasPermission(permission, null);
    }

    protected static boolean hasPermission(Permission permission, String id) {
        return permission.getPermissionHandler().hasPermission(id);
    }
}
