package com.xebialabs.deployit.core.rest.api;

import com.google.common.base.Strings;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.core.service.PaginationService;
import com.xebialabs.deployit.core.service.ReportGenerator;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.Application;
import com.xebialabs.deployit.plugin.api.udm.Environment;
import com.xebialabs.deployit.repository.ConfigurationItemData;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.SearchParameters;
import com.xebialabs.deployit.security.PermissionDeniedException;
import com.xebialabs.deployit.security.permission.PermissionHelper;
import com.xebialabs.deployit.task.TaskMetadata;
import com.xebialabs.deployit.task.TaskType;
import com.xebialabs.deployit.task.archive.TaskArchive;
import com.xebialabs.deployit.task.archive.TaskArchiveQueue;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.List;
import java.util.Map;

import static com.xebialabs.deployit.task.TaskMetadata.*;

public class AbstractTaskRestrictedResource extends AbstractSecuredResource {
    @Autowired
    protected TaskArchive taskArchive;

    @Autowired
    protected TaskArchiveQueue taskArchiveQueue;

    @Autowired
    protected PaginationService paginationService;

    @Autowired
    protected RepositoryService repositoryService;

    @Autowired
    protected ReportGenerator reportGenerator;

    protected void checkAccessTo(Map<String, String> taskMetadata) {
        if (PermissionHelper.isCurrentUserAdmin()) return;
        boolean isControlTask = TaskType.CONTROL.name().equals(taskMetadata.get(TaskMetadata.TASK_TYPE));
        boolean isCloudTask = taskMetadata.get(CLOUD_OPERATION) != null;
        if (isCloudTask) {
            doCloudTaskPermissionCheck(taskMetadata);
        } else if (isControlTask) {
            doControlTaskPermissionCheck(taskMetadata);
        } else {
            doDeploymentTaskPermissionCheck(taskMetadata);
        }
    }

    protected boolean hasTaskPermission(Map<String, String> taskMetadata) {
        boolean isControlTask = TaskType.CONTROL.name().equals(taskMetadata.get(TaskMetadata.TASK_TYPE));
        boolean isCloudTask = taskMetadata.get(CLOUD_OPERATION) != null;
        if (isCloudTask) {
            return hasCloudTaskPermission(taskMetadata);
        } else if (isControlTask) {
            return hasTargetOrIsAdmin(taskMetadata) && hasControlTaskViewPermission(taskMetadata);
        } else {
            return hasDeploymentTaskPermission(taskMetadata, getApplicationId(taskMetadata));
        }
    }

    private void doDeploymentTaskPermissionCheck(Map<String, String> taskMetadata) {
        String appId = getApplicationId(taskMetadata);
        if (appId == null) {
            throw new DeployitException("Unable to locate application for report");
        }

        if (!hasDeploymentTaskPermission(taskMetadata, appId)) {
            throw PermissionDeniedException.withMessage(
                    "You need read permissions for both the environment and application to view the archived task.");
        }
    }

    private boolean hasDeploymentTaskPermission(final Map<String, String> taskMetadata, final String appId) {
        final String environmentId = getEnvironmentId(taskMetadata);
        return appId != null && hasReadPermission(environmentId) && hasReadPermission(appId);
    }

    private String getApplicationId(final Map<String, String> taskMetadata) {
        if (taskMetadata.containsKey(APPLICATION_INTERNAL_ID)) {
            Integer internalAppId = Integer.parseInt(taskMetadata.get(APPLICATION_INTERNAL_ID));
            return getCiIdByInternalId(internalAppId, Application.class);
        }
        return getApplicationByName(taskMetadata.get(APPLICATION));
    }

    private String getEnvironmentId(final Map<String, String> taskMetadata) {
        if (taskMetadata.containsKey(ENVIRONMENT_INTERNAL_ID)) {
            return getCiIdByInternalId(Integer.parseInt(taskMetadata.get(ENVIRONMENT_INTERNAL_ID)), Environment.class);
        }
        return taskMetadata.get(ENVIRONMENT_ID);
    }

    private String getApplicationByName(String name) {
        final SearchParameters criteria = new SearchParameters().setType(Type.valueOf(Application.class)).setName(name);
        return searchCiIdByCriteria(criteria);
    }

    private String getCiIdByInternalId(Integer internalId, Class ciType) {
        final SearchParameters criteria = new SearchParameters().setType(Type.valueOf(ciType)).setInternalId(internalId);
        return searchCiIdByCriteria(criteria);
    }

    private String searchCiIdByCriteria(SearchParameters criteria) {
        List<ConfigurationItemData> cis = repositoryService.list(criteria);
        return cis.size() != 1 ? null : cis.get(0).getId();
    }

    private void doControlTaskPermissionCheck(Map<String, String> taskMetadata) {
        if (!hasTargetOrIsAdmin(taskMetadata)) {
            throw PermissionDeniedException.withMessage(
                    "Control tasks that don't have their target CI archived can only be viewed by admin.");
        }
        if (!hasControlTaskViewPermission(taskMetadata)) {
            throw PermissionDeniedException.withMessage(
                    "You need read permission on an archived control task's target CI to view it.");
        }
    }

    private boolean hasTargetOrIsAdmin(final Map<String, String> taskMetadata) {
        return !Strings.isNullOrEmpty(taskMetadata.get(TaskMetadata.CONTROL_TASK_TARGET_CI)) || PermissionHelper.isCurrentUserAdmin();
    }

    private boolean hasControlTaskViewPermission(final Map<String, String> taskMetadata) {
        return hasReadPermission(taskMetadata.get(TaskMetadata.CONTROL_TASK_TARGET_CI));
    }

    private boolean hasCloudTaskPermission(Map<String, String> taskMetadata) {
        String cloudEnvId = taskMetadata.get(CLOUD_ENVIRONMENT_ID);
        String cloudTemplId = taskMetadata.get(CLOUD_ENVIRONMENT_TEMPLATE_ID);
        return hasReadPermission(cloudEnvId) && hasReadPermission(cloudTemplId);
    }

    private void doCloudTaskPermissionCheck(Map<String, String> taskMetadata) {
        if (!hasCloudTaskPermission(taskMetadata)) {
            throw PermissionDeniedException.withMessage(
                    "You need read permissions for both the environment and the environment template to view the archived cloud task.");
        }
    }

    public void setPaginationService(PaginationService paginationService) {
        this.paginationService = paginationService;
    }

    public void setReportGenerator(ReportGenerator reportGenerator) {
        this.reportGenerator = reportGenerator;
    }
}
