package com.xebialabs.deployit.core.rest.api;

import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.checksum.ChecksumAlgorithmProvider;
import com.xebialabs.deployit.core.config.ClientProperties;
import com.xebialabs.deployit.core.defaults.CiDefaultEntityPropertiesProcessor;
import com.xebialabs.deployit.core.rest.resteasy.Workdir;
import com.xebialabs.deployit.core.rest.resteasy.WorkdirHolder;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.core.rest.util.RepositoryHelper;
import com.xebialabs.deployit.engine.api.RepositoryService;
import com.xebialabs.deployit.engine.api.dto.ArtifactAndData;
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemProperties;
import com.xebialabs.deployit.engine.spi.command.*;
import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.io.ArtifactFileUtils;
import com.xebialabs.deployit.plugin.api.reflect.DescriptorRegistry;
import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.PropertyKind;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.artifact.ArchiveArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.Artifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.FolderArtifact;
import com.xebialabs.deployit.plugin.api.udm.artifact.SourceArtifact;
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem;
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage;
import com.xebialabs.deployit.repository.SearchParameters;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.security.PermissionDeniedException;
import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.deployit.security.permission.PermissionHelper;
import com.xebialabs.deployit.service.externalproperties.validation.ExternalPropertiesValidatorChain;
import com.xebialabs.deployit.service.validation.Validator;
import com.xebialabs.deployit.util.TFiles;
import com.xebialabs.overthere.local.LocalFile;
import com.xebialabs.xldeploy.packager.placeholders.SourceArtifactScanner;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.*;
import java.util.function.Predicate;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.core.rest.api.DtoReader.ciDataToCiId;
import static com.xebialabs.deployit.core.rest.api.SearchParametersFactory.createSearchParams;
import static com.xebialabs.deployit.core.rest.api.SearchParametersFactory.createSearchParamsForCandidateValues;
import static com.xebialabs.deployit.plugin.api.reflect.PropertyKind.*;
import static com.xebialabs.deployit.repository.WorkDirFactory.ARTIFACT_WORKDIR_PREFIX;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.EDIT_REPO;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.READ;
import static java.util.Collections.singletonList;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

public class RepositoryResource extends AbstractSecuredResource implements RepositoryService {

    private static final Logger logger = LoggerFactory.getLogger(RepositoryResource.class);

    private final RoleService roleService;

    private final com.xebialabs.deployit.repository.RepositoryService repositoryService;

    private final Validator validator;

    private final ExternalPropertiesValidatorChain externalPropertiesValidatorChain;

    private final RepositoryHelper repositoryHelper;

    private final CiDefaultEntityPropertiesProcessor ciDefaultEntityPropertiesProcessor;

    private final SourceArtifactScanner sourceArtifactScanner;

    @Autowired
    private ClientProperties clientProperties;

    private final ChecksumAlgorithmProvider checksumAlgorithmProvider;

    public RepositoryResource(com.xebialabs.deployit.repository.RepositoryService repositoryService,
                              RoleService roleService,
                              Validator validator,
                              SourceArtifactScanner sourceArtifactScanner,
                              RepositoryHelper repositoryHelper,
                              CiDefaultEntityPropertiesProcessor ciDefaultEntityPropertiesProcessor,
                              ExternalPropertiesValidatorChain externalPropertiesValidatorChain,
                              ChecksumAlgorithmProvider checksumAlgorithmProvider) {
        this.roleService = roleService;
        this.repositoryService = repositoryService;
        this.validator = validator;
        this.sourceArtifactScanner = sourceArtifactScanner;
        this.repositoryHelper = repositoryHelper;
        this.ciDefaultEntityPropertiesProcessor = ciDefaultEntityPropertiesProcessor;
        this.externalPropertiesValidatorChain = externalPropertiesValidatorChain;
        this.checksumAlgorithmProvider = checksumAlgorithmProvider;
    }

    @Override
    public ConfigurationItem construct(Type type) {
        // Requires only LOGIN permission.
        ConfigurationItem item = DescriptorRegistry.getDescriptor(type).newInstance("");
        ciDefaultEntityPropertiesProcessor.fillDefaultEntities(item);
        return item;
    }

    @Override
    public ConfigurationItem read(String id) {
        checkReadPermission(id);
        return repositoryService.read(id, 1, null, true, false);
    }

    @Override
    public List<ConfigurationItem> read(List<String> ids) {
        Map<String, Boolean> permissionsMap = PermissionHelper.hasPermission(READ, ids);
        Predicate<String> canRead = input -> (permissionsMap.containsKey(input) && permissionsMap.get(input)) || roleService.isReadOnlyAdmin();
        List<String> idsToRead = ids.stream().filter(canRead).collect(toList());
        return repositoryService.read(idsToRead, 1, true);
    }

    @Override
    public List<ConfigurationItemId> query(Type type, String parent, String ancestor, String namePattern,
                                           DateTime lastModifiedBefore, DateTime lastModifiedAfter,
                                           long page, long resultPerPage) {
        return queryV2(type, parent, ancestor, namePattern, null, lastModifiedBefore, lastModifiedAfter, page, resultPerPage);
    }

    @Override
    public List<ConfigurationItemId> queryV2(Type type, String parent, String ancestor, String namePattern, String idPattern,
                                             DateTime lastModifiedBefore, DateTime lastModifiedAfter,
                                             long page, long resultPerPage) {
        return list(createSearchParams(type, page, resultPerPage, parent, ancestor, namePattern, idPattern,
                lastModifiedBefore != null ? lastModifiedBefore.toGregorianCalendar() : null,
                lastModifiedAfter != null ? lastModifiedAfter.toGregorianCalendar() : null,
                0));
    }

    @Override
    public List<ConfigurationItemId> queryV3(Type type, String parent, String ancestor, String namePattern, String idPattern,
                                             DateTime lastModifiedBefore, DateTime lastModifiedAfter,
                                             long page, long resultPerPage, ConfigurationItemProperties properties) {
        return list(createSearchParams(type, page, resultPerPage, parent, ancestor, namePattern, idPattern,
                lastModifiedBefore != null ? lastModifiedBefore.toGregorianCalendar() : null,
                lastModifiedAfter != null ? lastModifiedAfter.toGregorianCalendar() : null,
                0, properties));
    }

    @Override
    public List<ConfigurationItemId> candidateValues(String propertyName, String namePattern, String idPattern,
                                                     long page, long resultPerPage, ConfigurationItem ci) {
        return list(createSearchParamsForCandidateValues(propertyName, namePattern, idPattern, page, resultPerPage, ci));
    }

    private List<ConfigurationItemId> list(final SearchParameters searchParams) {
        if (!roleService.isReadOnlyAdmin()) {
            READ.getPermissionHandler().applyPermission(searchParams);
        }
        return repositoryService.list(searchParams).stream().map(ciDataToCiId).collect(toList());
    }

    @Override
    public ConfigurationItem create(final String id, final ConfigurationItem ci) {
        checkPermission(EDIT_REPO, id);
        Checks.checkArgument(id.equals(ci.getId()),
                "The Configuration item id is [%s], but the id parameter is [%s]", ci.getId(), id);
        return createInternal(ci);
    }

    @Override
    @Workdir(prefix = ARTIFACT_WORKDIR_PREFIX)
    public ConfigurationItem create(final String id, final ArtifactAndData aad) {
        WorkDir workDir = WorkdirHolder.get();
        checkPermission(EDIT_REPO, id);
        Artifact artifact = aad.getArtifact();
        Checks.checkArgument(id.equals(artifact.getId()), "The Configuration item id is [%s], but the id parameter is [%s]", artifact.getId(), id);
        LocalFile localFile = readArtifactData(aad, workDir);
        artifact.setFile(localFile);
        if (artifact instanceof FolderArtifact) {
            Checks.checkTrue(TFiles.isArchive(artifact.getFile().getPath()),
                    "The selected artifact [%s] is not a folder", artifact.getFile().getName());
        }

        if(artifact instanceof ArchiveArtifact) {
            Checks.checkTrue(TFiles.isArchive(artifact.getFile().getPath()),
                    "The selected artifact [%s] is not a ZIP archive", artifact.getFile().getName());
        }

        ConfigurationItem ci = createInternal(artifact);
        localFile.deleteRecursively();
        return ci;
    }

    private ConfigurationItem createInternal(final ConfigurationItem ci) {
        if (ci instanceof SourceArtifact) {
            ArtifactFileUtils.handleArtifact(sourceArtifactScanner, (SourceArtifact) ci, checksumAlgorithmProvider);
        }

        validator.validateCi(ci);
        List<ValidationMessage> validationMessages = validator.validateInputHint(ci);
        validateExternalProperties(ci, validationMessages);
        List<String> ids = new ArrayList<>();
        repositoryHelper.checkReadAccessOnRelations(null, ci, ids, Collections.emptySet());
        if (!ids.isEmpty()) {
            throw new PermissionDeniedException("Permission READ is not granted on the following linked CIs: " + ids);
        }
        Set<ConfigurationItem> ciSet = new CiCollector(singletonList(ci)).collect();
        repositoryHelper.publishCommand(new CreateCisCommand(new ArrayList<>(ciSet)));
        if (repositoryService.exists(ci.getId())) {
            ConfigurationItem configurationItem = repositoryService.read(ci.getId(), 1);
            configurationItem.get$validationMessages().addAll(validationMessages);
            return configurationItem;
        } else {
            logger.info("CI Creator did not include [{}] on list of CIs to be created. Returning null.", ci.getId());
            return null;
        }
    }

    @Override
    @Workdir(prefix = ARTIFACT_WORKDIR_PREFIX)
    public List<ConfigurationItem> create(List<ConfigurationItem> cis) {
        Set<String> ciIds = cis.stream().map(ConfigurationItem::getId).collect(toSet());
        ciIds.stream().filter(ciId -> !ciIds.contains(parentId(ciId))).forEach(id -> checkPermission(EDIT_REPO, id));
        validator.validateCis(cis);

        List<String> ids = new ArrayList<>();
        for (ConfigurationItem ci : cis) {
            if (ci instanceof SourceArtifact) {
                ArtifactFileUtils.handleArtifact(sourceArtifactScanner, (SourceArtifact) ci, checksumAlgorithmProvider);
            }
            repositoryHelper.checkReadAccessOnRelations(null, ci, ids, ciIds);
        }
        if (!ids.isEmpty()) {
            throw new PermissionDeniedException("Permission READ is not granted on the following linked CIs: " + ids);
        }

        Set<ConfigurationItem> ciSet = new CiCollector(cis).collect();
        RepositoryBaseCommand event = new CreateCisCommand(new ArrayList<>(ciSet));
        repositoryHelper.publishCommand(event);
        return repositoryHelper.reloadEntities(cis);
    }

    @Override
    public ConfigurationItem update(String id, ConfigurationItem ci) {
        checkPermission(EDIT_REPO, id);
        Checks.checkArgument(id.equals(ci.getId()),
                "The Configuration item id is [%s], but the id parameter is [%s]", ci.getId(), id);
        return updateInternal(ci);
    }

    @Override
    @Workdir(prefix = ARTIFACT_WORKDIR_PREFIX)
    public ConfigurationItem update(final String id, final ArtifactAndData aad) {
        WorkDir workDir = WorkdirHolder.get();
        checkPermission(EDIT_REPO, id);
        Checks.checkArgument(id.equals(aad.getArtifact().getId()),
                "The Configuration item id is [%s], but the id parameter is [%s]", aad.getArtifact().getId(), id);
        LocalFile localFile = readArtifactData(aad, workDir);
        aad.getArtifact().setFile(localFile);
        ConfigurationItem ci = updateInternal(aad.getArtifact());
        localFile.deleteRecursively();
        return ci;
    }

    private ConfigurationItem updateInternal(final ConfigurationItem ci) {
        validator.validateCi(ci);
        List<ValidationMessage> validationMessages = validator.validateInputHint(ci);
        validateExternalProperties(ci, validationMessages);
        ConfigurationItem previous = repositoryService.read(ci.getId(), false);
        Checks.checkArgument(repositoryHelper.isFileUriNotChanged(ci, previous),
                "File URIs cannot be updated. If you want to change the file URI, create a new CI for the artifact. " +
                        "(https://docs.xebialabs.com/xl-deploy/how-to/add-an-externally-stored-artifact-to-a-package.html#changing-the-uri-of-a-deployable-artifact)");
        repositoryHelper.checkIfConvertibleType(ci, previous);
        List<String> ids = new ArrayList<>();
        repositoryHelper.checkReadAccessOnRelations(previous, ci, ids, Collections.emptySet());
        if (!ids.isEmpty()) {
            throw new PermissionDeniedException("Permission READ is not granted on the following linked CIs: " + ids);
        }
        repositoryHelper.checkIfUpdatedReadonlyProperty(ci, previous);

        repositoryHelper.publishCommand(new UpdateCiCommand(previous, ci));
        ConfigurationItem configurationItem = repositoryService.read(ci.getId(), 1, null, false);
        configurationItem.get$validationMessages().addAll(validationMessages);
        return configurationItem;
    }

    private class MultiUpdatePermissionCheck implements Predicate<ConfigurationItem> {
        private final Set<String> ciIds;
        private final Set<String> existingIds;
        private final Set<String> canNotEdit;
        private final Map<String, Boolean> permissionCache = new HashMap<>();

        private MultiUpdatePermissionCheck(final Set<String> ciIds, final Set<String> existingIds) {
            this.ciIds = ciIds;
            this.existingIds = existingIds;
            this.canNotEdit = this.existingIds.stream().filter(ciId -> !hasPermission(EDIT_REPO, ciId)).collect(toSet());
        }

        @Override
        public boolean test(final ConfigurationItem configurationItem) {
            String currentId = configurationItem.getId();
            while (ciIds.contains(parentId(currentId))) {
                if (canNotEdit.contains(currentId)) return false;
                if (existingIds.contains(currentId)) return true;
                currentId = parentId(currentId);
            }
            return permissionCache.computeIfAbsent(currentId, id -> hasPermission(EDIT_REPO, id));
        }
    }

    @Override
    public List<ConfigurationItem> update(List<ConfigurationItem> cis) {
        // FIXME: should we check the repo#edit permission and validate the CIs here?
        Set<String> ciIds = cis.stream().map(ConfigurationItem::getId).collect(toSet());
        Set<String> existingIds = ciIds.stream().filter(repositoryService::exists).collect(toSet());
        List<ConfigurationItem> filteredCis = cis
                .stream()
                .filter(new MultiUpdatePermissionCheck(ciIds, existingIds))
                .collect(toList());
        return repositoryHelper.createOrUpdateAndReloadCis(filteredCis, Optional.of(existingIds));
    }

    @Override
    public ConfigurationItem move(String id, String targetId) {
        checkPermission(EDIT_REPO, id);
        checkPermission(EDIT_REPO, targetId);
        String idRoot = id.split("/")[0];
        String targetRoot = targetId.split("/")[0];
        checkArgument(idRoot.equals(targetRoot),
                "Cannot move ci [%s] to [%s] because that is outside of the root where it is stored.", id, targetId);
        repositoryHelper.publishCommand(new MoveCiCommand(id, targetId));
        return read(targetId);
    }

    @Override
    public ConfigurationItem copy(String id, String targetId) {
        checkPermission(EDIT_REPO, id);
        checkPermission(EDIT_REPO, targetId);
        String idRoot = id.split("/")[0];
        String targetRoot = targetId.split("/")[0];
        checkArgument(idRoot.equals(targetRoot),
                "Cannot copy ci [%s] to [%s] because that is outside of the root where it is stored.", id, targetId);
        repositoryHelper.publishCommand(new CopyCiCommand(id, targetId, read(id).getType()));
        return read(targetId);
    }

    @Override
    public ConfigurationItem rename(String id, String targetName) {
        checkPermission(EDIT_REPO, id);
        repositoryHelper.publishCommand(new RenameCiCommand(id, targetName, read(id).getType()));
        String targetId = parentId(id) + '/' + targetName;
        return read(targetId);
    }

    private String parentId(final String id) {
        return id.substring(0, id.lastIndexOf('/'));
    }

    @Override
    public List<ConfigurationItem> validate(final List<ConfigurationItem> cis) {
        validator.validateCis(cis);
        for (ConfigurationItem ci : cis) {
            List<ValidationMessage> validationMessages = validator.validateInputHint(ci);
            validateExternalProperties(ci, validationMessages);
            ci.get$validationMessages().addAll(validationMessages);
        }
        return cis;
    }

    @Override
    public void delete(String id) {
        checkDeletePermission(id);
        repositoryHelper.publishCommand(new DeleteCiCommand(id));
    }

    @Override
    public void deleteList(List<String> ids) {
        repositoryHelper.publishCommand(new DeleteCisCommand(ids));
    }

    protected void checkDeletePermission(String id) {
        checkPermission(EDIT_REPO, id);
    }

    @Override
    public Boolean exists(String id) {
        return repositoryService.exists(id);
    }

    @Override
    public Boolean isSecure(String id) {
        ConfigurationItem ci = repositoryService.read(id);
        return ci != null && ci.get$internalId().equals(ci.get$securedCi());
    }

    private LocalFile readArtifactData(ArtifactAndData aad, WorkDir workDir) {
        checkArgument(aad.getFilename() != null, "The filename for the artifact should not be null");
        final LocalFile localFile = workDir.newFile(aad.getFilename());
        try (OutputStream outputStream = localFile.getOutputStream();
             InputStream inputStream = aad.getDataInputStream()) {
            copy(inputStream, outputStream);
        } catch (IOException e) {
            String message = "Could not write Artifact data for [" + aad.getArtifact().getId() + "] to [" + localFile + "]";
            logger.error(message, e);
            throw new DeployitException(message);
        }
        return localFile;
    }

    static class CiCollector {
        private Collection<? extends ConfigurationItem> cis;

        public CiCollector(Collection<? extends ConfigurationItem> cis) {
            this.cis = cis;
        }

        public Set<ConfigurationItem> collect() {
            Set<ConfigurationItem> collector = new HashSet<>();
            collectNestedCis(cis, collector);
            return collector;
        }

        private Set<ConfigurationItem> collectNestedCis(Collection<? extends ConfigurationItem> cis,
                                                        Set<ConfigurationItem> collector) {
            for (ConfigurationItem ci : cis) {
                collector.add(ci);
                for (PropertyDescriptor property : ci.getType().getDescriptor().getPropertyDescriptors()) {
                    PropertyKind kind = property.getKind();
                    if (kind == SET_OF_CI || kind == LIST_OF_CI) {
                        Collection<ConfigurationItem> children = getChildren(ci, property);
                        collectNestedCis(children, collector);
                    } else if (kind == CI) {
                        Collection<ConfigurationItem> children = getChildrenForCI(ci, property);
                        collectNestedCis(children, collector);
                    }
                }
            }
            return collector;
        }

        private Collection<ConfigurationItem> getChildrenForCI(ConfigurationItem parent, PropertyDescriptor property) {
            // Single CI reference with 'asContainment' on child
            ConfigurationItem ci = (ConfigurationItem) property.get(parent);
            List<ConfigurationItem> children = new ArrayList<>();
            if (ci != null && isChild(ci, parent)) {
                children.add(ci);
            }
            return children;
        }

        private Collection<ConfigurationItem> getChildren(ConfigurationItem parent, PropertyDescriptor property) {

            @SuppressWarnings("unchecked")
            Collection<ConfigurationItem> references = (Collection<ConfigurationItem>) property.get(parent);

            // Parent/child relationship is modeled on the parent
            if (property.isAsContainment()) {
                return references;
            }

            // Parent/child relationship is modeled on the child
            List<ConfigurationItem> children = new ArrayList<>();
            for (ConfigurationItem ci : references) {
                if (isChild(ci, parent)) {
                    children.add(ci);
                }
            }

            return children;
        }

        private boolean isChild(ConfigurationItem ci, ConfigurationItem parent) {
            for (PropertyDescriptor property : ci.getType().getDescriptor().getPropertyDescriptors()) {
                PropertyKind kind = property.getKind();
                if (kind == CI && property.isAsContainment() && parent.equals(property.get(ci))) {
                    return true;
                }
            }
            return false;
        }
    }

    private void copy(InputStream from, OutputStream to) throws IOException {
        byte[] buf = new byte[clientProperties.uploadFileBufferSize()];
        while (true) {
            int r = from.read(buf);
            if (r == -1) {
                break;
            }
            to.write(buf, 0, r);
        }
    }

    private void validateExternalProperties(ConfigurationItem ci, List<ValidationMessage> validationMessages) {
        if (ci instanceof BaseConfigurationItem) {
            validationMessages.addAll(externalPropertiesValidatorChain.validate((BaseConfigurationItem) ci));
        }
    }
}
