package com.xebialabs.deployit.core.rest.xml;

import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.xebialabs.deployit.core.xml.PasswordEncryptingXStreamCiConverter;
import com.xebialabs.deployit.engine.api.dto.ValidatedConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.repository.RepositoryAdapterFactory;
import com.xebialabs.xltype.serialization.CiReference;
import com.xebialabs.xltype.serialization.ConfigurationItemConverter;
import com.xebialabs.xltype.serialization.util.ReferenceUtils;
import com.xebialabs.xltype.serialization.xstream.CiXstreamReader;
import com.xebialabs.xltype.serialization.xstream.XStreamProvider;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Map;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;

@Component
@XStreamProvider(tagName = "configuration-item", readable = ConfigurationItem.class)
public class XStreamCiConverterWithRepository extends PasswordEncryptingXStreamCiConverter {

    static final ThreadLocal<Map<String, ConfigurationItem>> DESERIALIZATION_CONTEXT = ThreadLocal.withInitial(() -> newHashMap());

    static final ThreadLocal<List<CiReference>> DESERIALIZATION_REFERENCES = ThreadLocal.withInitial(() -> newArrayList());

    static final ThreadLocal<RepositoryAdapterFactory> REPOSITORY_ADAPTER_FACTORY = new ThreadLocal<>();

    private final RepositoryAdapterFactory factory;

    @Autowired
    public XStreamCiConverterWithRepository(RepositoryAdapterFactory factory) {
        this.factory = factory;
    }

    @SuppressWarnings("rawtypes")
    @Override
    public Object unmarshal(HierarchicalStreamReader reader, UnmarshallingContext context) {

        // This is the same as super.unmarshal()...
        ConfigurationItemConverter converter = createConverter(context);
        ConfigurationItem ci = converter.readCi(new CiXstreamReader(reader));

        // .. but now we have to capture the references
        DESERIALIZATION_REFERENCES.get().addAll(converter.getReferences());

        REPOSITORY_ADAPTER_FACTORY.set(factory);

        if (ci instanceof ValidatedConfigurationItem) {
            DESERIALIZATION_CONTEXT.get().put(ci.getId(), ((ValidatedConfigurationItem) ci).getWrapped());
        } else {
            DESERIALIZATION_CONTEXT.get().put(ci.getId(), ci);
        }

        return ci;
    }

    static void postProcess() {
        try {
            RepositoryAdapterFactory factory = REPOSITORY_ADAPTER_FACTORY.get();
            ReferenceUtils.resolveReferences(DESERIALIZATION_REFERENCES.get(),
                    DESERIALIZATION_CONTEXT.get(), factory != null ? factory.create() : null);
        } finally {
            clear();
        }
    }

    public static void clear() {
        DESERIALIZATION_REFERENCES.remove();
        DESERIALIZATION_CONTEXT.remove();
        REPOSITORY_ADAPTER_FACTORY.remove();
    }

}
