package com.xebialabs.deployit.core.rest.api;

import com.xebialabs.deployit.engine.api.dto.Deployment;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plugin.api.udm.DeployedApplication;
import com.xebialabs.deployit.plugin.api.udm.Environment;
import com.xebialabs.deployit.security.permission.Permission;
import org.springframework.stereotype.Component;

import java.util.List;

import static com.xebialabs.deployit.checks.Checks.checkArgument;
import static com.xebialabs.deployit.core.rest.api.PermissionChecker.HasPermission.hasPermission;
import static com.xebialabs.deployit.security.permission.DeployitPermissions.*;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.READ;

@Component
class PermissionChecker {

    public void checkPermission(Deployment deployment, List<List<Deployment>> dependencies) {
        checkPermission(deployment);
        checkApplicationPermissions(dependencies, hasPermission());
        dependencies.forEach(depl ->
                depl.forEach(d -> {
                    try {
                        checkPermission(d);
                    } catch (InvalidDeploymentException e) {
                        StringBuilder errorMsgBuilder = new StringBuilder(e.getMessage());
                        errorMsgBuilder.append(String.format(
                                "The error occurred while deploying dependencies for application \"%s\"", deployment
                                        .getDeployedApplication().getName()));
                        throw new InvalidDeploymentException(errorMsgBuilder.toString());
                    }
                }));
    }

    private void checkApplicationPermissions(List<List<Deployment>> allDeployments, HasPermission f) {
        allDeployments.stream().forEach(deployments -> deployments.stream().forEach(deployment -> {
            DeployedApplication deployedApplication = checkAndCast(deployment.getDeployedApplication(), DeployedApplication.class);
            String applicationId = deployedApplication.getVersion().getApplication().getId();
            if (!f.check(READ, applicationId)) {
                throw new InvalidDeploymentException(String.format("Unable to satisfy application dependencies. Application \"%s\" not found.", deployedApplication.getName()));
            }
        }));
    }

    void checkPermission(Deployment deployment) {
        Permission deploymentPermission;
        switch (deployment.getDeploymentType()) {
            case INITIAL:
                deploymentPermission = DEPLOY_INITIAL;
                break;
            case UPDATE:
                deploymentPermission = DEPLOY_UPGRADE;
                break;
            case UNDEPLOYMENT:
                deploymentPermission = UNDEPLOY;
                break;
            default:
                throw new InvalidDeploymentException(String.format("Unable to determine deployment type \"%s\"", deployment.getDeploymentType()));
        }
        DeployedApplication deployedApplication = (DeployedApplication) deployment.getDeployedApplication();
        try {
            checkPermission(deploymentPermission, deployedApplication.getEnvironment());
        } catch (InvalidDeploymentException e) {
            StringBuilder errorMsgBuilder = new StringBuilder("You do not have the required permissions.");
            errorMsgBuilder.append(String.format(" Application \"%s\" requires \"%s\" permission on environment \"%s\".",
                    deployedApplication.getName(), deploymentPermission.getPermissionName(), deployedApplication
                            .getEnvironment().getName()));
            throw new InvalidDeploymentException(errorMsgBuilder.toString());
        }
    }

    void checkPermission(Permission permission, Environment environment) {
        String environmentID = environment.getId();
        if (!permission.getPermissionHandler().hasPermission(environmentID)) {
            String errorMsgBuilder = "You do not have the required permissions." +
                    String.format("\"%s\" permission is required on environment \"%s\".",
                            permission.getPermissionName(), environment.getName());
            throw new InvalidDeploymentException(errorMsgBuilder);
        }
    }

    private <T extends ConfigurationItem> T checkAndCast(ConfigurationItem ci, Class<T> clazz) {
        checkArgument(clazz.isAssignableFrom(ci.getClass()), "%s is not a %s", ci, clazz.getSimpleName());
        return clazz.cast(ci);
    }

    @FunctionalInterface
    interface HasPermission {
        boolean check(Permission permission, String ciId);

        static HasPermission hasPermission() {
            return (p, ciId) -> p.getPermissionHandler().hasPermission(ciId);
        }

    }
}


