package com.xebialabs.deployit.core.rest.api;

import com.xebialabs.deployit.checks.Checks;
import com.xebialabs.deployit.core.rest.resteasy.Workdir;
import com.xebialabs.deployit.core.rest.resteasy.WorkdirHolder;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.engine.api.InspectionService;
import com.xebialabs.deployit.engine.api.dto.Inspection;
import com.xebialabs.deployit.engine.api.execution.StepState;
import com.xebialabs.deployit.engine.tasker.Task;
import com.xebialabs.deployit.engine.tasker.TaskExecutionEngine;
import com.xebialabs.deployit.engine.tasker.TaskNotFoundException;
import com.xebialabs.deployit.engine.tasker.TaskSpecification;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.WorkDir;
import com.xebialabs.deployit.security.Permissions;
import com.xebialabs.deployit.service.discovery.DiscoveryResult;
import com.xebialabs.deployit.service.discovery.DiscoveryService;
import com.xebialabs.deployit.spring.BeanWrapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.List;

import static com.xebialabs.deployit.core.rest.resteasy.Workdir.Clean.ONLY_ON_EXCEPTION;
import static com.xebialabs.deployit.repository.WorkDirFactory.TASK_WORKDIR_PREFIX;
import static com.xebialabs.deployit.security.permission.DeployitPermissions.DISCOVERY;

/**
 */
@Service
public class DiscoveryResource extends AbstractSecuredResource implements InspectionService {

    @Autowired
    private BeanWrapper<DiscoveryService> discoveryService;

    @Autowired
    private RepositoryService repositoryService;

    @Autowired
    private BeanWrapper<TaskExecutionEngine> engine;

    @Override
    public Inspection prepare(ConfigurationItem configurationItem) {
        checkPermission(DISCOVERY);
        return new Inspection(configurationItem);
    }

    @Override
    public Inspection prepare(String id) {
        checkPermission(DISCOVERY);
        ConfigurationItem read = repositoryService.read(id);
        return new Inspection(read);
    }

    @Override
    @Workdir(prefix = TASK_WORKDIR_PREFIX, clean = ONLY_ON_EXCEPTION)
    public String createTask(final Inspection inspection) {
        WorkDir workDir = WorkdirHolder.get();
        checkPermission(DISCOVERY);
        TaskSpecification taskSpecification = discoveryService.get().prepareDiscovery(inspection.getConfigurationItem(), workDir);
        logger.info("Creating inspection task for [{}]", inspection.getConfigurationItem());
        return engine.get().register(taskSpecification);
    }

    @Override
    public List<ConfigurationItem> retrieveInspectionResults(String taskId) {
        checkPermission(DISCOVERY);
        logger.info("Retrieving discovery results for task [{}]", taskId);
        try {
            Task task = engine.get().retrieve(taskId);
            logger.warn("Attempt to retrieve discovery results from a still-running task");
            int currentStepNr = task.getCurrentStepNr();
            StepState currentStep = task.getStep(currentStepNr);
            logger.warn("task [{}], state [{}], current step [{}/{}], description [{}], log: [{}]", taskId,
                    task.getState(), currentStepNr, task.getNrSteps(),
                    currentStep.getDescription(), currentStep.getLog());
            throw new Checks.ConditionFailedException("Task [%s] is not yet finished.", taskId);
        } catch (TaskNotFoundException tnfe) {
            // Ok!
        }
        DiscoveryResult discoveryResults = discoveryService.get().getDiscoveryResults(taskId);
        if (!discoveryResults.getOwner().equals(Permissions.getAuthenticatedUserName())) {
            throw new Checks.ConditionFailedException("You were not the owner of task [%s]", taskId);
        }
        return discoveryResults.getItems();
    }

    private static final Logger logger = LoggerFactory.getLogger(DiscoveryResource.class);
}
