package com.xebialabs.deployit.core.rest.api;

import com.xebialabs.deployit.checksum.ChecksumAlgorithmProvider;
import com.xebialabs.deployit.core.defaults.CiDefaultEntityPropertiesProcessor;
import com.xebialabs.deployit.core.ordering.ConfigurationItemIdComparator;
import com.xebialabs.deployit.core.rest.api.support.PaginationSupport;
import com.xebialabs.deployit.core.rest.util.RepositoryHelper;
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemProperties;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.SearchParameters;
import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.deployit.service.externalproperties.validation.ExternalPropertiesValidatorChain;
import com.xebialabs.deployit.service.validation.Validator;
import com.xebialabs.xldeploy.packager.placeholders.SourceArtifactScanner;
import org.jboss.resteasy.spi.HttpResponse;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import javax.ws.rs.core.Context;
import java.util.Collections;
import java.util.List;

import static com.xebialabs.deployit.security.permission.DeployitPermissions.IMPORT_REMOVE;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.EDIT_REPO;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.READ;

@Service
public class DeployitRepositoryResource extends RepositoryResource {

    @Context
    HttpResponse httpResponse;

    private final ConfigurationItemIdComparator comparator;

    private final RepositoryService repositoryService;

    @Autowired
    public DeployitRepositoryResource(final RepositoryService repositoryService,
                                      final RoleService roleService,
                                      final Validator validator,
                                      final ConfigurationItemIdComparator comparator,
                                      final SourceArtifactScanner scanner,
                                      final RepositoryHelper repositoryHelper,
                                      final CiDefaultEntityPropertiesProcessor ciDefaultEntityPropertiesProcessor,
                                      final ExternalPropertiesValidatorChain externalPropertiesValidatorChain,
                                      final ChecksumAlgorithmProvider checksumAlgorithmProvider) {
        super(repositoryService, roleService, validator, scanner, repositoryHelper, ciDefaultEntityPropertiesProcessor, externalPropertiesValidatorChain, checksumAlgorithmProvider);
        this.comparator = comparator;
        this.repositoryService = repositoryService;
    }

    @Override
    protected void checkDeletePermission(String id) {
        if (IMPORT_REMOVE.isApplicableTo(id)) {
            checkPermission(IMPORT_REMOVE, id);
        } else {
            checkPermission(EDIT_REPO, id);
        }
    }

    @Override
    public List<ConfigurationItemId> queryV2(final Type type, final String parent, final String ancestor, final String namePattern, final String idPattern, final DateTime lastModifiedBefore, final DateTime lastModifiedAfter, final long page, final long resultPerPage) {
        List<ConfigurationItemId> result = super.queryV2(type, parent, ancestor, namePattern, idPattern, lastModifiedBefore, lastModifiedAfter, page, resultPerPage);
        if (isNoPagination(resultPerPage)) {
            sort(result);
        }
        return result;
    }

    @Override
    public List<ConfigurationItemId> queryV3(Type type, String parent, String ancestor, String namePattern, String idPattern, DateTime lastModifiedBefore, DateTime lastModifiedAfter, long page, long resultPerPage, ConfigurationItemProperties properties) {
        List<ConfigurationItemId> result = super.queryV3(type, parent, ancestor, namePattern, idPattern, lastModifiedBefore, lastModifiedAfter, page, resultPerPage, properties);
        if (isNoPagination(resultPerPage)) {
            sort(result);
        }
        long totalCount = queryCount(type, parent, ancestor, namePattern);
        PaginationSupport.addTotalCountHeader(totalCount, httpResponse);
        return result;
    }

    private long queryCount(Type type, String parent, String ancestor, String namePattern) {
        SearchParameters searchParams = new SearchParameters();
        searchParams.setType(type).setParent(parent).setAncestor(ancestor).setName(namePattern);
        READ.getPermissionHandler().applyPermission(searchParams);
        return repositoryService.count(searchParams);
    }

    private void sort(final List<ConfigurationItemId> result) {
        Collections.sort(result, comparator);
    }

    private boolean isNoPagination(final long resultPerPage) {
        return resultPerPage <= 0;
    }
}
