package com.xebialabs.deployit.core.rest.websockets

import com.xebialabs.deployit.ServerConfiguration
import com.xebialabs.deployit.core.rest.converters.{CiIdMessageConverter, CiIdsMessageConverter}
import com.xebialabs.deployit.core.rest.json.CiRefsJsonWriter
import org.eclipse.jetty.websocket.api.{WebSocketBehavior, WebSocketPolicy}
import org.springframework.beans.factory.annotation.{Autowired, Qualifier, Value}
import org.springframework.context.annotation.{Bean, Configuration}
import org.springframework.messaging.converter.MessageConverter
import org.springframework.messaging.simp.config.{ChannelRegistration, MessageBrokerRegistry}
import org.springframework.messaging.support.ChannelInterceptor
import org.springframework.scheduling.TaskScheduler
import org.springframework.web.socket.config.annotation.{EnableWebSocketMessageBroker, StompEndpointRegistry, WebSocketMessageBrokerConfigurer}
import org.springframework.web.socket.server.jetty.JettyRequestUpgradeStrategy
import org.springframework.web.socket.server.support.DefaultHandshakeHandler

import java.util.concurrent.TimeUnit
import java.util.{List => JList}
import com.xebialabs.deployit.core.config.{ClientConfiguration, WebSocketsConfiguration}

@Configuration
@EnableWebSocketMessageBroker
class WebSocketMessageBrokerConfig(@Autowired ciRefsJsonWriter: CiRefsJsonWriter,
                                   @Autowired @Qualifier("websocketsScheduler") taskScheduler: TaskScheduler,
                                   @Autowired @Qualifier("userInterceptor") userInterceptor: ChannelInterceptor,
                                   @Autowired webSocketsConfiguration: WebSocketsConfiguration,
                                   @Autowired clientConfiguration: ClientConfiguration
                                  ) extends WebSocketMessageBrokerConfigurer {

  private val inputBufferSize: Int = webSocketsConfiguration.messageBroker.inputBufferSize

  override def configureMessageBroker(registry: MessageBrokerRegistry): Unit = {
    registry.setApplicationDestinationPrefixes("/app")
    registry.enableSimpleBroker("/topic").setTaskScheduler(taskScheduler)
  }

  @Bean
  def handshakeHandler(): DefaultHandshakeHandler = {
    val policy = new WebSocketPolicy(WebSocketBehavior.SERVER)
    policy.setInputBufferSize(inputBufferSize)
    val idleTimeout = TimeUnit.MINUTES.toMillis(clientConfiguration.getSession.getTimeoutMinute)
    policy.setIdleTimeout(idleTimeout)
    new DefaultHandshakeHandler(new JettyRequestUpgradeStrategy(policy))
  }

  override def registerStompEndpoints(registry: StompEndpointRegistry): Unit = {
    registry
      .addEndpoint("/ws")
      .setAllowedOriginPatterns("*")
      .setHandshakeHandler(handshakeHandler())
      .withSockJS()
  }

  override def configureClientInboundChannel(registry: ChannelRegistration): Unit = {
    registry.interceptors(userInterceptor)
  }

  override def configureMessageConverters(messageConverters: JList[MessageConverter]): Boolean = {
    messageConverters.add(new CiIdsMessageConverter(ciRefsJsonWriter))
    true
  }
}
