package com.xebialabs.deployit.core.rest.api

import com.xebialabs.deployit.core.api.HostService
import com.xebialabs.deployit.core.service.PaginationService
import com.xebialabs.deployit.engine.api.dto.{ConfigurationItemId, DeploymentInfo, Ordering, Paging}
import com.xebialabs.deployit.repository.{DeployedApplicationsRepository, EnvironmentRepository}
import org.jboss.resteasy.spi.HttpResponse
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service

import java.{lang, util}
import javax.ws.rs.core.Context
import scala.jdk.CollectionConverters._

@Service
class HostServiceImpl @Autowired()(environmentRepository: EnvironmentRepository,
                                   deployedApplicationsRepository: DeployedApplicationsRepository,
                                   paginationService: PaginationService,
                                   repositoryResource: RepositoryResource) extends HostService {

  @Context val response: HttpResponse = null

  private def checkIfUserHasPermissions(hostId: String) = repositoryResource.read(hostId)

  override def findDeployedApplications(hostId: String,
                                        deployedAppName: String,
                                        paging: Paging,
                                        order: Ordering): util.List[ConfigurationItemId] = {
    checkIfUserHasPermissions(hostId)
    val limited = paginationService.getLimitedPaging(paging)
    paginationService.addPagingHeaderIfNeeded(paging, limited, paginationService.toSetHeader(response), () => deployedApplicationsRepository.countByHost(hostId, deployedAppName))
    deployedApplicationsRepository.findByHost(hostId, deployedAppName, paging, order).asJava
  }

  override def countEnvironmentsByHost(hostId: String, environmentName: String): lang.Integer = {
    checkIfUserHasPermissions(hostId)
    environmentRepository.count(hostId, Option(environmentName))
  }

  override def getEnvironmentsByHost(hostId: String, environmentName: String, paging: Paging, order: Ordering): util.List[DeploymentInfo] = {
    checkIfUserHasPermissions(hostId)
    val limited = paginationService.getLimitedPaging(paging)
    paginationService.addPagingHeaderIfNeeded(paging, limited, paginationService.toSetHeader(response), () => environmentRepository.count(hostId, Option(environmentName)))
    environmentRepository.listEnvironmentsByHost(hostId, Option(environmentName), paging, order)
  }

}
