package com.xebialabs.deployit.core.rest.api.reports;

import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.xebialabs.deployit.core.api.dto.Report;
import com.xebialabs.deployit.core.api.dto.ReportLine;
import com.xebialabs.deployit.engine.api.dto.ConfigurationItemId;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.FilterType;
import com.xebialabs.deployit.task.archive.TaskArchive;

import java.util.List;
import java.util.Map;

public class AggregatedDeploymentsKeyIndicator extends TaskArchiveReport {

    public AggregatedDeploymentsKeyIndicator(TaskArchive taskArchive) {
        super(taskArchive);
    }

    public Report report(final FilterType filterType, List<ConfigurationItemId> filterCriteria) {
        final LoadingCache<String, TaskStateData> tasks = CacheBuilder.newBuilder().build(CacheLoader.from((String input) -> new TaskStateData()));

        for (ConfigurationItemId appOrEnvName : filterCriteria) {
            ArchivedTaskSearchParameters searchTasks = cloneSearchParameters().forFilterType(filterType, appOrEnvName);

            taskArchive.searchForMaps(searchTasks, true).forEach(task ->
                tasks.getUnchecked(getEntityId(task, filterType)).addTaskAndDuration(task)
            );
        }

        return generateReport(tasks.asMap());
    }

    private static Report generateReport(Map<String, TaskStateData> tasks) {
        final Report report = new Report();
        for (Map.Entry<String, TaskStateData> entry : tasks.entrySet()) {
            final ReportLine line = report.addLine();
            TaskStateData task = entry.getValue();
            line.addValue("entityName", entry.getKey());
            line.addValue("noOfSuccessfulDeployments", task.getNoOfSuccessfulDeployments().toString());
            line.addValue("noOfFailedDeployments", task.getNoOfFailedDeployments().toString());
            line.addValue("noOfAbortedDeployments", task.getNoOfAbortedDeployments().toString());
            line.addValue("noOfRollbacks", task.getNoOfRollbacks().toString());
            line.addValue("averageDuration", DeploymentsKeyIndicator.getAverageDeploymentTime(task));
        }
        return report;
    }

    private static String getEntityId(Map<String, Object> task, FilterType filterType) {
        switch (filterType) {
        case APPLICATION:
            return (String) task.get("main_application");
        case ENVIRONMENT:
            return (String) task.get("environment");
        default:
            throw new IllegalArgumentException("Unsupported filter type: " + filterType);
        }
    }
}
