package com.xebialabs.deployit.core.rest.api

import java.{lang, util}

import com.xebialabs.deployit.core.api.dto.{AddUserCredentials, IdDto, UpdateUserCredentials, UserCredentialsView}
import com.xebialabs.deployit.core.api.{XldUserCredentialsService => XldUserCredentialsServiceInterface}
import com.xebialabs.deployit.security.model.XldUserCredentials
import com.xebialabs.deployit.security.permission.PlatformPermissions.{EDIT_SECURITY, VIEW_SECURITY}
import com.xebialabs.deployit.security.service.UserCredentialsService
import org.springframework.stereotype.Controller

import scala.jdk.CollectionConverters._

object XldUserCredentialsService {
  def mapToUserCredentialsView(credentials: XldUserCredentials): UserCredentialsView = {
    new UserCredentialsView(
      credentials.id,
      credentials.label,
      credentials.username,
      credentials.email,
      credentials.default,
      credentials.password.orNull,
      credentials.passphrase.orNull,
      credentials.privateKey.orNull
    )
  }
}

@Controller
class XldUserCredentialsServiceImpl(userCredentialsService: UserCredentialsService) extends AbstractUserRestrictedResource with XldUserCredentialsServiceInterface {

  override def listUserCredentials(username: String): util.Set[UserCredentialsView] = {
    checkCurrentUserOrPermissions(username, VIEW_SECURITY, EDIT_SECURITY)
    userCredentialsService.findByProfileUsername(username).map(
      credentials => XldUserCredentialsService.mapToUserCredentialsView(credentials)
    ).asJava
  }

  override def getDefaultUserCredentials(username: String): UserCredentialsView = {
    checkCurrentUserOrPermissions(username, VIEW_SECURITY, EDIT_SECURITY)
    XldUserCredentialsService.mapToUserCredentialsView(userCredentialsService.findDefaultCredentialsByProfileUsername(username))
  }

  override def saveUserCredentials(username: String, credentials: AddUserCredentials): IdDto[lang.Long] = {
    checkCurrentUserOrPermission(username, EDIT_SECURITY)
    new IdDto(userCredentialsService.saveCredentials(
      new XldUserCredentials(
        null,
        username,
        credentials.getLabel,
        credentials.getUsername,
        credentials.getEmail,
        credentials.getDefaultCredentials,
        Option(credentials.getPassword),
        Option(credentials.getPassphrase),
        Option(credentials.getPrivateKey)
      )))
  }

  override def updateUserCredentials(username: String, credentials: UpdateUserCredentials): IdDto[lang.Long] = {
    checkCurrentUserOrPermission(username, EDIT_SECURITY)
    new IdDto(userCredentialsService.updateCredentials(new XldUserCredentials(
      credentials.getId,
      username,
      credentials.getLabel,
      credentials.getUsername,
      credentials.getEmail,
      credentials.getDefaultCredentials,
      Option(credentials.getPassword),
      Option(credentials.getPassphrase),
      Option(credentials.getPrivateKey)
    )))
  }
}

