package com.xebialabs.deployit.core.rest.api.reports;

import java.io.*;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;
import java.util.stream.Stream;
import org.apache.poi.ss.usermodel.Workbook;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.joda.time.LocalDateTime;
import org.joda.time.Period;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.joda.time.format.PeriodFormatter;
import org.joda.time.format.PeriodFormatterBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.deployit.ReleaseInfo;
import com.xebialabs.deployit.ServerConfiguration;
import com.xebialabs.deployit.core.api.dto.Report;
import com.xebialabs.deployit.core.api.dto.ReportLine;
import com.xebialabs.deployit.core.rest.api.DownloadResource;
import com.xebialabs.deployit.core.rest.api.reports.mapper.ReportToCsvMapper;
import com.xebialabs.deployit.core.rest.resteasy.WorkdirHolder;
import com.xebialabs.deployit.report.audit.AuditPermissionRoleRow;
import com.xebialabs.deployit.report.audit.RolePrincipalPermissionRow;

import static java.lang.String.format;

public class ReportUtils {
    private static final Logger log = LoggerFactory.getLogger(ReportUtils.class);

    public static final DateTimeFormatter DATE_FORMAT = DateTimeFormat.forPattern("yyyyMMdd");
    static final PeriodFormatter HH_MM_SS = new PeriodFormatterBuilder().minimumPrintedDigits(2).printZeroAlways().appendHours().appendSeparator(":").appendMinutes().appendSeparator(":").appendSeconds().toFormatter();

    private ReportUtils() {
    }

    static String formatToHoursMinsSecs(long millis) {
        Period period = new Duration(millis).toPeriod();
        return HH_MM_SS.print(period);
    }

    public static Stream<String> toCsv(Set<String> fields, Stream<ReportLine> report) {
        return new ReportToCsvMapper(report, fields).getCsv();
    }


    public static List<DateTime> getMonthsOfInterval(DateTime begin, DateTime end) {
        List<DateTime> result = new ArrayList<>();
        DateTime current = begin;
        while (current.isBefore(end) || current.isEqual(begin)) {
            result.add(current);
            current = current.plusMonths(1);
        }
        if (result.size() == 1 && begin.getMonthOfYear() != end.getMonthOfYear()) {
            result.add(end);
        }
        return result;
    }

    public static String createDownloadToken(DownloadResource downloadResource, Stream<ReportLine> reportLines, String fileName) {
        return createDownloadToken(downloadResource, null, reportLines, fileName);
    }

    public static String createDownloadToken(DownloadResource downloadResource, Set<String> firstRowFields, Stream<ReportLine> reportLines, String fileName) {
        try {
            File to = new File(WorkdirHolder.get().getPath(), fileName);
            try (BufferedWriter writer = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(to), Charset.defaultCharset()));
                 Stream<String> s = toCsv(firstRowFields, reportLines)
            ) {
                s.forEachOrdered(line -> {
                    try {
                        writer.write(line);
                        writer.write('\n');
                    } catch (IOException io) {
                        throw new RuntimeException(io);
                    }
                });
            }
            return downloadResource.register(to, "text/csv");
        } catch (Exception re) {
            log.error("Oops!", re);
            WorkdirHolder.get().delete();
            throw new IllegalStateException(re);
        }
    }

    public static String createDownloadToken(DownloadResource downloadResource, Report report, String fileName) {
        return createDownloadToken(downloadResource, report.getLines().stream(), fileName);
    }

    public static String createAuditReport(DownloadResource downloadResource, List<RolePrincipalPermissionRow> globalAuditReport,
                                           List<AuditPermissionRoleRow> folderAuditReport) {
        final DateTimeFormatter auditDateFormat = DateTimeFormat.forPattern("yyyy-MM-dd_hhmmss");
        final String serverName = ServerConfiguration.getInstance().getServerUrl();
        final String deployVersion = ReleaseInfo.getReleaseInfo().getVersion();
        return createAuditReportToken(AuditReportUtility.createAuditReport(globalAuditReport, folderAuditReport, serverName, deployVersion), downloadResource, auditDateFormat);
    }

    public static String createAuditReportToken(Workbook workbook, DownloadResource downloadResource, DateTimeFormatter auditDateFormat) {
        String fileName = format("%s - Deploy.xlsx", auditDateFormat.print(LocalDateTime.now()));
        File to = new File(WorkdirHolder.get().getPath(), fileName);
        try (FileOutputStream fos = new FileOutputStream(to)) {
            workbook.write(fos);
        } catch (IOException e) {
            log.error(e.getMessage(), e);
        }
        return downloadResource.register(to, "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet.main+xml");
    }
}
