package com.xebialabs.deployit.core.service;

import com.google.common.collect.Sets;
import com.xebialabs.deployit.core.api.resteasy.Date;
import com.xebialabs.deployit.core.rest.api.DownloadResource;
import com.xebialabs.deployit.core.rest.api.reports.ReportUtils;
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource;
import com.xebialabs.deployit.core.util.TaskFilterUtils;
import com.xebialabs.deployit.engine.api.dto.*;
import com.xebialabs.deployit.engine.api.execution.FetchMode;
import com.xebialabs.deployit.report.audit.AuditPreviewReport;
import com.xebialabs.deployit.report.audit.AuditReportRepository;
import com.xebialabs.deployit.report.audit.AuditPermissionRoleRow;
import com.xebialabs.deployit.report.audit.RolePrincipalPermissionRow;
import com.xebialabs.deployit.security.RoleService;
import com.xebialabs.deployit.security.permission.PermissionHelper;
import com.xebialabs.deployit.security.permission.PlatformPermissions;
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters;
import com.xebialabs.deployit.task.FilterType;
import com.xebialabs.deployit.task.TaskMetadata;
import com.xebialabs.deployit.task.TaskType;
import com.xebialabs.deployit.task.archive.TaskArchive;
import com.xebialabs.deployit.task.archive.TaskReader;
import com.xebialabs.deployit.task.archive.sql.CachingTaskReader;
import com.xebialabs.deployit.task.archive.sql.schema.ArchivedControlTasks;
import com.xebialabs.deployit.task.archive.sql.schema.ArchivedDeploymentTasks;
import com.xebialabs.deployit.task.archive.sql.schema.ArchivedTasks;
import com.xebialabs.deployit.task.archive.sql.schema.ArchivedTasksShared;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.StringReader;
import java.util.*;
import java.util.stream.Stream;

import static com.xebialabs.deployit.checks.Checks.checkNotNull;
import static com.xebialabs.deployit.core.util.TypeConversions.enforceBoolean;
import static com.xebialabs.deployit.core.rest.api.reports.ReportUtils.DATE_FORMAT;
import static com.xebialabs.deployit.security.permission.DeployitPermissions.REPORT_VIEW;
import static java.lang.String.format;

@Service
public class ReportGenerator extends AbstractSecuredResource {
    private final TaskArchive taskArchive;
    private final AuditReportRepository sqlAuditReportRepository;
    private final RoleService roleService;

    @Autowired
    public ReportGenerator(TaskArchive taskArchive, AuditReportRepository sqlAuditReportRepository, RoleService roleService) {
        this.taskArchive = taskArchive;
        this.sqlAuditReportRepository = sqlAuditReportRepository;
        this.roleService = roleService;
    }

    public static final Set<String> DEPLOYMENT_ALLOWED_SORT_FIELDS = new HashSet<>(Arrays.asList(
            "id", "failures", "owner", "state", "startDate", "completionDate", TaskMetadata.APPLICATION, TaskMetadata.ENVIRONMENT_ID, TaskMetadata.TASK_TYPE, "worker_name"));

    public static final Set<String> CONTROL_ALLOWED_SORT_FIELDS = new HashSet<>(Arrays.asList(
            "id", "failures", "owner", "state", "startDate", "completionDate", "description", TaskMetadata.CONTROL_TASK_TARGET_CI, TaskMetadata.TASK_NAME, "worker_name"));

    public static final EnumSet<TaskType> ALLOWED_TASK_TYPE_FILTER_VALUES = EnumSet.of(
            TaskType.INITIAL,
            TaskType.UPGRADE,
            TaskType.UNDEPLOY,
            TaskType.ROLLBACK,
            TaskType.INSPECTION,
            TaskType.CONTROL
    );

    public ArchivedTaskSearchParameters buildFilteredTaskSearchParameters(Date begin, Date end, Paging paging, List<Ordering> order,
                                                                          String filterType, List<String> users,
                                                                          List<String> taskStates, List<String> taskTypes,
                                                                          FetchMode fetchMode, String taskId,
                                                                          List<ConfigurationItemId> configurationItemIds,
                                                                          Set<String> allowedSortFields, boolean onlySuccessful, String workerName) {
        checkNotNull(begin, "begin");
        checkNotNull(end, "end");
        EnumSet<TaskType> types = EnumSet.noneOf(TaskType.class);
        types.addAll(TaskFilterUtils.buildTaskTypesFilter(taskTypes, onlySuccessful));
        TaskFilterUtils.validateTaskTypesFilter(types, ALLOWED_TASK_TYPE_FILTER_VALUES);

        ArchivedTaskSearchParameters searchParameters = buildTaskSearchParameters(paging, order, types, allowedSortFields);
        searchParameters.createdBetween(begin.asDateTime(), end.asDateTime());
        searchParameters.forSuccessfulOnly(onlySuccessful);
        searchParameters.withFetchMode(fetchMode);
        searchParameters.withUniqueId(taskId);
        searchParameters.withWorkerName(workerName);

        if (filterType != null) {
            FilterType filter = FilterType.valueOf(filterType.toUpperCase());
            if (filter != FilterType.NONE) {
                searchParameters.forFilterType(filter, configurationItemIds);
            }
        }
        if (!users.isEmpty()) {
            searchParameters.forUsers(Sets.newHashSet(users));
        }
        if (!taskStates.isEmpty()) {
            searchParameters.withTaskStates(taskStates);
        }
        if (shouldApplyReadPermission()) {
            searchParameters.applySecurityParameters(PlatformPermissions.READ.getPermissionHandler().getRolesPermissionsPair());
        }

        return searchParameters;
    }

    private boolean shouldApplyReadPermission() {
        return !(PermissionHelper.isCurrentUserAdmin() || roleService.isReadOnlyAdmin());
    }

    public ArchivedTaskSearchParameters buildTaskSearchParameters(Paging paging, List<Ordering> order, EnumSet<TaskType> taskTypes, Set<String> allowedSortFields) {
        paging.validate();
        for (Ordering ordering : order) {
            ordering.validate(allowedSortFields);
        }

        ArchivedTaskSearchParameters searchParameters = new ArchivedTaskSearchParameters();
        searchParameters.thatAreOfType(taskTypes);
        if (order.isEmpty()) {
            searchParameters.orderBy("startDate", false);
        } else {
            for (Ordering ordering : order) {
                searchParameters.orderBy(ordering.field(), ordering.isAscending());
            }
        }
        if (shouldApplyReadPermission()) {
            searchParameters.applySecurityParameters(PlatformPermissions.READ.getPermissionHandler().getRolesPermissionsPair());
        }

        searchParameters.showPage(paging.page(), paging.resultsPerPage());

        return searchParameters;
    }

    public Stream<DeploymentTaskReportLine> streamTaskReport(ArchivedTaskSearchParameters searchParameters) {
        checkPermission(REPORT_VIEW);
        return taskArchive.searchForMaps(searchParameters, true).map(this::asDeploymentTaskReportLine);
    }

    public Stream<ControlTaskReportLine> streamControlTasksReport(ArchivedTaskSearchParameters searchParameters) {
        checkPermission(PlatformPermissions.REPORT_VIEW);
        return taskArchive
                .searchForMaps(searchParameters, true)
                .map(this::asControlTaskReportLine);
    }

    public Map<Integer, List<String>> getEnvironments() {
        checkPermission(REPORT_VIEW);
        return taskArchive.getAllArchivedEnvironmentsWithIds();
    }

    public Map<Integer, List<String>> getApplications() {
        checkPermission(REPORT_VIEW);
        return taskArchive.getAllArchivedApplicationsWithIds();
    }

    private DeploymentTaskReportLine asDeploymentTaskReportLine(final Map<String, Object> map) {
        DeploymentTaskReportLine line = new DeploymentTaskReportLine();
        line.setTaskId((String) map.get(ArchivedTasksShared.task_id().name()));
        line.setStartDate((DateTime) map.get(ArchivedTasksShared.start_date().name()));
        line.setCompletionDate((DateTime) map.get(ArchivedDeploymentTasks.end_date().name()));
        line.setStatus((String) map.get(ArchivedDeploymentTasks.status().name()));
        line.setTaskType(transformTaskType(TaskType.valueOf((String) map.get(ArchivedDeploymentTasks.task_type().name()))));
        line.setUser((String) map.get(ArchivedDeploymentTasks.owner().name()));
        String environment = (String) map.get(ArchivedDeploymentTasks.environment().name());
        line.setEnvironment(environment.substring(environment.lastIndexOf('/') + 1));
        line.setEnvironmentId(environment);
        line.setEnvironmentIdWithoutRoot(environment.substring(environment.indexOf('/') + 1));
        line.setDeploymentPackage((String) map.get(ArchivedDeploymentTasks.packages().name()));
        line.setRolledBack(enforceBoolean(map.get(ArchivedDeploymentTasks.rolled_back().name())));
        if (map.containsKey(ArchivedTasks.task_details().name())) {
            TaskReader reader = new CachingTaskReader(new StringReader((String) map.get(ArchivedTasks.task_details().name())));
            line.setBlock(reader.withoutSteps().getBlock());
        }
        line.setWorkerName((String) map.get(ArchivedTasks.worker_name().name()));
        return line;
    }

    private ControlTaskReportLine asControlTaskReportLine(final Map<String, Object> map) {
        ControlTaskReportLine line = new ControlTaskReportLine();
        line.setTaskId((String) map.get(ArchivedControlTasks.task_id().name()));
        line.setStarted((DateTime) map.get(ArchivedControlTasks.start_date().name()));
        line.setFinished((DateTime) map.get(ArchivedControlTasks.end_date().name()));
        line.setState((String) map.get(ArchivedControlTasks.status().name()));
        line.setOwner((String) map.get(ArchivedControlTasks.owner().name()));
        line.setTaskName((String) map.get(ArchivedControlTasks.control_task_name().name()));
        line.setControlTaskTargetCI((String) map.get(ArchivedControlTasks.target_ci().name()));
        line.setDescription((String) map.get(ArchivedControlTasks.description().name()));
        if (map.containsKey(ArchivedTasks.task_details().name())) {
            TaskReader reader = new CachingTaskReader(new StringReader((String) map.get(ArchivedTasks.task_details().name())));
            line.setBlock(reader.withoutSteps().getBlock());
        }
        line.setWorkerName((String) map.get(ArchivedTasks.worker_name().name()));
        return line;
    }

    public DateTime safeDateTime(final Date date, final DateTime defaultValue) {
        return date == null ? defaultValue : date.asDateTime();
    }

    public String buildControlTaskReportFileName(final Date begin, final Date end) {
        DateTime theBeginningOfTime = new DateTime(1, 1, 1, 0, 0, 0);
        DateTime beginDT = safeDateTime(begin, theBeginningOfTime);
        DateTime endDT = safeDateTime(end, new DateTime());
        return format("tasks-%s-%s.csv", DATE_FORMAT.print(beginDT), DATE_FORMAT.print(endDT));
    }

    private String transformTaskType(TaskType taskType) {
        switch (taskType) {
            case INITIAL:
                return "Initial";
            case UPGRADE:
                return "Update";
            case UNDEPLOY:
                return "Undeployment";
            case ROLLBACK:
                return "Rollback";
            default:
                return "Unknown";
        }
    }

    public String generateAuditReport(List<String> folders, DownloadResource downloadResource) {
        checkPermission(REPORT_VIEW);
        List<RolePrincipalPermissionRow> globalAuditReport = sqlAuditReportRepository.getGlobalAuditReport();
        List<AuditPermissionRoleRow> folderAuditReport = sqlAuditReportRepository.getAuditReport(folders);
        return ReportUtils.createAuditReport(downloadResource, globalAuditReport, folderAuditReport);
    }

    public AuditPreviewReport previewAuditReport(List<String> folders, List<Ordering> order, Paging paging) {
        checkPermission(REPORT_VIEW);
        return sqlAuditReportRepository.previewAuditReport(folders, order, paging);
    }
}
