package com.xebialabs.deployit.core.rest.api

import com.google.common.collect.Sets
import com.xebialabs.deployit.core.api.dto.ReportLine
import com.xebialabs.deployit.core.api.resteasy.Date
import com.xebialabs.deployit.core.converters.{ControlTaskReportLineConverter, ControlTaskReportLineField, DeploymentReportLineField, DeploymentTaskReportLineConverter}
import com.xebialabs.deployit.core.rest.api.reports.ReportUtils
import com.xebialabs.deployit.core.rest.api.reports.ReportUtils.DATE_FORMAT
import com.xebialabs.deployit.core.rest.resteasy.Workdir
import com.xebialabs.deployit.core.rest.resteasy.Workdir.Clean.ALWAYS
import com.xebialabs.deployit.core.rest.secured.AbstractSecuredResource
import com.xebialabs.deployit.core.service.ReportGenerator._
import com.xebialabs.deployit.core.service.{PaginationService, ReportGenerator}
import com.xebialabs.deployit.core.util.TaskFilterUtils
import com.xebialabs.deployit.engine.api.dto._
import com.xebialabs.deployit.engine.api.execution.FetchMode
import com.xebialabs.deployit.engine.api.{ReportService, dto}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.{Application, Environment}
import com.xebialabs.deployit.repository.WorkDirFactory.{DOWNLOAD_WORKDIR_PREFIX, PREVIEW_WORKDIR_PREFIX}
import com.xebialabs.deployit.repository.sql.{CiRepository, ConfigurationItemDataWithInternalId}
import com.xebialabs.deployit.security.permission.PlatformPermissions.{READ, REPORT_VIEW}
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters
import com.xebialabs.deployit.task.TaskType.CONTROL
import com.xebialabs.deployit.task.archive.TaskArchive
import grizzled.slf4j.Logging
import org.jboss.resteasy.spi.HttpResponse
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Controller
import org.springframework.transaction.annotation.Transactional

import java.lang.String.format
import java.util.EnumSet.of
import java.util.stream.{Stream => JStream}
import java.util.{Collections, List => JList, Set => JSet}
import javax.ws.rs.core.Context
import scala.jdk.CollectionConverters._
import scala.language.implicitConversions

@Controller
@Autowired
class ReportServiceImpl(reportGenerator: ReportGenerator,
                        paginationService: PaginationService,
                        ciRepository: CiRepository,
                        taskArchive: TaskArchive,
                        downloadResource: DownloadResource) extends AbstractSecuredResource with ReportService with Logging {
  @Context
  private var response: HttpResponse = _

  def setResponse(response: HttpResponse): Unit = {
    this.response = response
  }

  private def setCiFilters(searchParameters: ArchivedTaskSearchParameters,
                           configurationItemIds: JList[ConfigurationItemId]): Unit = {
    val (applications, environments) = TaskFilterUtils.parseCiFilters(Option(configurationItemIds.asScala).toList.flatten)
    applications.foreach(searchParameters.forApplication)
    environments.foreach(searchParameters.forEnvironment)
  }

  private def streamReport[T](pagingMaybe: Option[Paging],
                              buildParams: Paging => ArchivedTaskSearchParameters,
                              stream: ArchivedTaskSearchParameters => JStream[T]): JStream[T] = {
    val limitedMaybe = pagingMaybe.map(paging => (paging, paginationService.getLimitedPaging(paging)))
    val searchParameters = buildParams(limitedMaybe.map(_._2).getOrElse(new Paging()))
    limitedMaybe
      .foreach( limitedPaging =>
        paginationService.addPagingHeaderIfNeeded(
          limitedPaging._1,
          limitedPaging._2,
          paginationService.toSetHeader(response),
          () => taskArchive.countTotalResults(searchParameters))
      )
    stream(searchParameters)
  }

  private def streamDeploymentTaskReport(begin: Date,
                                         end: Date,
                                         paging: Option[Paging],
                                         order: JList[dto.Ordering],
                                         users: JList[String],
                                         taskStates: JList[String],
                                         taskTypes: JList[String],
                                         fetchMode: FetchMode,
                                         taskId: String,
                                         onlySuccessful: Boolean,
                                         workerName: String,
                                         configurationItemIds: JList[ConfigurationItemId]) = {
    streamReport(
      paging,
      limited => {
        val params = reportGenerator.buildFilteredTaskSearchParameters(begin, end, limited, order, null, users, taskStates,
          taskTypes, fetchMode, taskId, Nil.asJava, DEPLOYMENT_ALLOWED_SORT_FIELDS, onlySuccessful, workerName)
        setCiFilters(params, configurationItemIds)
        params
      },
      reportGenerator.streamTaskReport
    )
  }

  override def getTaskReport(begin: Date,
                             end: Date,
                             paging: Paging,
                             order: JList[dto.Ordering],
                             users: JList[String],
                             taskStates: JList[String],
                             taskTypes: JList[String],
                             fetchMode: FetchMode,
                             taskId: String,
                             onlySuccessful: Boolean,
                             workerName: String,
                             configurationItemIds: JList[ConfigurationItemId]): JStream[DeploymentTaskReportLine] = {
    streamDeploymentTaskReport(begin, end, Option(paging), order, users, taskStates, taskTypes, fetchMode, taskId,
      onlySuccessful, workerName, configurationItemIds)
  }

  @Workdir(prefix = DOWNLOAD_WORKDIR_PREFIX, clean = ALWAYS)
  @Transactional(value = "reportingTransactionManager", readOnly = true)
  override def downloadTaskReport(begin: Date,
                                  end: Date,
                                  order: JList[dto.Ordering],
                                  users: JList[String],
                                  taskStates: JList[String],
                                  taskTypes: JList[String],
                                  taskId: String,
                                  onlySuccessful: Boolean,
                                  workerName: String,
                                  configurationItemIds: JList[ConfigurationItemId]): String = {
    val stream = streamDeploymentTaskReport(begin, end, None, order, users, taskStates, taskTypes, FetchMode.SUMMARY,
      taskId, onlySuccessful, workerName, configurationItemIds).map[ReportLine](DeploymentTaskReportLineConverter.toReportLine)
    val fileName = format("tasks-%s-%s.csv", DATE_FORMAT.print(begin.asDateTime), DATE_FORMAT.print(end.asDateTime))

    ReportUtils.createDownloadToken(downloadResource, DeploymentReportLineField.allFields.asJava, stream, fileName)
  }

  private def streamControlTaskReport(begin: Date,
                                      end: Date,
                                      states: JList[String],
                                      taskName: String,
                                      fetchMode: FetchMode,
                                      users: JList[String],
                                      order: JList[dto.Ordering],
                                      workerName: String,
                                      taskId: String,
                                      paging: Option[Paging],
                                      targets: JSet[String]) = {
    streamReport(
      paging,
      limited => {
        val params = reportGenerator.buildTaskSearchParameters(limited, order, of(CONTROL), CONTROL_ALLOWED_SORT_FIELDS)
        params.withTargets(targets)
        params.withTaskStates(states)
        params.withTaskName(taskName)
        params.withFetchMode(fetchMode)
        params.withWorkerName(workerName)
        params.inDateTimeRange(reportGenerator.safeDateTime(begin, null), reportGenerator.safeDateTime(end, null))
        params.withUniqueId(taskId)
        if (users != null && !users.isEmpty) {
          params.forUsers(Sets.newHashSet(users))
        }
        params
      },
      reportGenerator.streamControlTasksReport
    )
  }

  override def getControlTasksReport(begin: Date,
                                     end: Date,
                                     states: JList[String],
                                     taskName: String,
                                     fetchMode: FetchMode,
                                     users: JList[String],
                                     order: JList[dto.Ordering],
                                     workerName: String,
                                     taskId: String,
                                     paging: Paging,
                                     targets: JSet[String]): JStream[ControlTaskReportLine] = {
    streamControlTaskReport(begin, end, states, taskName, fetchMode, users, order, workerName, taskId, Option(paging), targets)
  }

  @Workdir(prefix = DOWNLOAD_WORKDIR_PREFIX, clean = ALWAYS)
  @Transactional(value = "reportingTransactionManager", readOnly = true)
  override def downloadControlTasksReport(begin: Date,
                                          end: Date,
                                          states: JList[String],
                                          taskName: String,
                                          users: JList[String],
                                          order: JList[dto.Ordering],
                                          workerName: String,
                                          taskId: String,
                                          targets: JSet[String]): String = {
    val report = streamControlTaskReport(begin, end, states, taskName, FetchMode.SUMMARY, users, order, workerName, taskId, None, targets)
      .map[ReportLine](ControlTaskReportLineConverter.toReportLine)
    ReportUtils.createDownloadToken(downloadResource, ControlTaskReportLineField.allFields.asJava, report, reportGenerator.buildControlTaskReportFileName(begin, end))
  }

  private def listAllCisOfTypeGrouped(ciType: Type,
                                      getLabel: ConfigurationItemDataWithInternalId => String) = {
    val params = SearchParametersFactory.createSearchParams(ciType, 0, -1)
    READ.getPermissionHandler.applyPermission(params)
    ciRepository.listInternal(params).asScala.map(item => item.internalId -> getLabel(item)).toMap
  }

  private def listTypePaths(archivedCis: Map[Integer, JList[String]],
                            getLabel: ConfigurationItemDataWithInternalId => String,
                            ciType: Type): JList[ConfigurationItemId] = {
    checkPermission(REPORT_VIEW)
    val cisInRepository: Map[Integer, String] = listAllCisOfTypeGrouped(ciType, getLabel)

    val matchedIds = archivedCis.keySet.intersect(cisInRepository.keySet)
    val onlyInArchive = archivedCis.flatMap { case (id, paths) =>
      if (matchedIds.contains(id)) paths.asScala.filter(path => cisInRepository(id) != path) else Nil
    }.toList

    val inArchiveAndRepository = cisInRepository.values.toList ::: onlyInArchive
    val result = if (hasPermission(REPORT_VIEW)) {
      inArchiveAndRepository ::: archivedCis.view.filterKeys(id => !cisInRepository.keySet.contains(id)).values.flatMap(_.asScala).toList
    } else inArchiveAndRepository
    result
      .distinct
      .sorted
      .map(new ConfigurationItemId(_, ciType))
      .asJava
  }

  override def listEnvironments(): JList[ConfigurationItemId] =
    listTypePaths(reportGenerator.getEnvironments.asScala.toMap, _.id, Type.valueOf(classOf[Environment]))

  override def listApplications(): JList[ConfigurationItemId] =
    listTypePaths(reportGenerator.getApplications.asScala.toMap, _.name, Type.valueOf(classOf[Application]))

  @Workdir(prefix = DOWNLOAD_WORKDIR_PREFIX, clean = ALWAYS)
  @Transactional(value = "reportingTransactionManager", readOnly = true)
  override def downloadAuditReport(): String =
    reportGenerator.generateAuditReport(Collections.emptyList(), downloadResource)

  @Workdir(prefix = DOWNLOAD_WORKDIR_PREFIX, clean = ALWAYS)
  @Transactional(value = "reportingTransactionManager", readOnly = true)
  override def downloadAuditReport(folders: JList[String]): String =
    reportGenerator.generateAuditReport(folders, downloadResource)

  @Workdir(prefix = PREVIEW_WORKDIR_PREFIX)
  @Transactional(value = "reportingTransactionManager", readOnly = true)
  override def previewAuditReport(folders: java.util.List[String], order: JList[dto.Ordering],
                                  paging: Paging): JStream[AuditPreviewRow] = {
    val previewReport = reportGenerator.previewAuditReport(folders, order, paging)
    val limited = paginationService.getLimitedPaging(paging)
    paginationService.addPagingHeaderIfNeeded(paging, limited, paginationService.toSetHeader(response), () => previewReport.total)
    previewReport.currentPageData.stream()
  }
}
