package com.xebialabs.deployit.provision
package steps

import com.xebialabs.deployit.plugin.api.flow.{ExecutionContext, StepExitCode}
import com.xebialabs.deployit.plugin.api.rules.{RulePostConstruct, Scope, StepMetadata, StepPostConstructContext}
import com.xebialabs.deployit.provision.resolver.TemplateResolver.ResolutionResult
import com.xebialabs.deployit.provision.resolver.{ProvisionedResolver, TemplateResolver}
import com.xebialabs.deployit.repository.core.Directory

import java.util.NoSuchElementException
import scala.util.{Failure, Success, Try}

@StepMetadata(name = "validate-provisioned-cis")
class ValidationStep extends ProvisionedItemsStep {

  override def validScope: Scope = Scope.PRE_PLAN

  @transient
  private[steps] lazy val templateResolver = TemplateResolver(deployedApplication, ignoreContextPlaceholders = true)

  @transient
  private[steps] lazy val provisionedResolver = ProvisionedResolver(deployedApplication)

  @RulePostConstruct
  def validateAndSetParameters(ctx: StepPostConstructContext): Unit = {
    validate(ctx)
    if (description == null || description.isEmpty) {
      description = "Validate configuration item templates"
    }
    order = if (order != null) order else 10
    setParameters(ctx)
  }

  override def execute(ctx: ExecutionContext): StepExitCode = {
    resolveDeployeds()
    val resolvedCIs = templateResolver.resolveBoundTemplate(deploymentPackage).toList ++ deployeds.flatMap(p => templateResolver.resolveBoundTemplate(p).toList)
    val resolvedCiIds = resolvedCIs.map(_.instanceId)
    val failed = validateBoundConfigurationItems(ctx, resolvedCIs) ||
      validateDirectories(ctx, resolvedCIs) ||
      validateDuplicateIds(ctx, resolvedCiIds) ||
      validateInvalidChars(ctx, resolvedCiIds)
    if (failed) {
      StepExitCode.FAIL
    } else {
      StepExitCode.SUCCESS
    }
  }

  private def resolveDeployeds(): Unit = {
    deployeds.foreach(p => provisionedResolver.resolveProvisioned(p))
  }

  private[steps] def validateDirectories(ctx: ExecutionContext, resolvedCis: List[ResolutionResult]): Boolean = {
    val directoryPath: String = deployedApplication.getEnvironment.getDirectoryPath
    if (directoryPath != null) {
      val allRoots = resolvedCis
        .map(_.instanceId.map(_.split("/").head))
        .filter(_.isSuccess)
        .map(_.getOrElse(throw new IllegalArgumentException("Cannot resolve CIs")))
        .distinct

      val lazyDirIdsToBeCreated = resolvedCis
        .filter(r => r.configurationItem.isSuccess)
        .map(_.configurationItem.get)
        .filter(c => c.getType.instanceOf(typeOf[Directory]))
        .map(_.getId).toSet

      !allRoots.forall { root =>
        val dirPath = s"$root/$directoryPath"
        ctx.logOutput(s"Validating directory $dirPath")
        val exists = repositoryService.exists(dirPath) || lazyDirIdsToBeCreated.contains(dirPath)
        if (!exists) {
          ctx.logError(s"$dirPath does not exist")
        } else {
          ctx.logOutput(s"$dirPath exists")
        }
        exists
      }
    } else {
      false
    }
  }

  private[steps] def validateBoundConfigurationItems(ctx: ExecutionContext, resolvedCis: List[ResolutionResult]): Boolean = {
    val idsWithExists = resolvedCis.map(r => {
      ctx.logOutput(s"Validating bound configuration item ${r.templateId}")
      r.templateId -> r.instanceId.map(id => id -> repositoryService.exists(id))
    })
    idsWithExists.foldLeft(false) { (b, tuple) => {
      tuple match {
        case (_, Success((id, true))) =>
          ctx.logError(s"Configuration item with ID $id already exists in the repository")
          true
        case (templateId, Failure(ex)) =>
          ctx.logError(s"Skipping validation of template $templateId as it was not resolved: ${ex.getMessage}")
          b
        case (_, Success((_, false))) =>
          b
      }
    }
    }
  }

  private[steps] def validateDuplicateIds(ctx: ExecutionContext, resolvedCis: List[Try[String]]): Boolean = {
    val by = resolvedCis.filter(_.isSuccess).map({ t =>
      val id = t.getOrElse(throw new NoSuchElementException("Cannot validate configuration item"))
      ctx.logOutput(s"Validating configuration item $id for duplication")
      id
    }).groupBy(id => id)
    val duplicateIds = by.filter(pair => pair._2.length > 1)
    duplicateIds.foreach {
      case (id, _) =>
        ctx.logError(s"$id is duplicated")
    }
    duplicateIds.nonEmpty
  }

  private[steps] def validateInvalidChars(ctx: ExecutionContext, resolvedCis: List[Try[String]]): Boolean = {
    val invalidChars = "%:[]*|\t\r\n"

    val findInvalid = findInvalidChar(invalidChars)(_)

    def charToString(char: Char) = {
      if (char.isWhitespace) {
        "whitespace"
      } else {
        char.toString
      }
    }

    resolvedCis.filter(_.isSuccess).map(_.getOrElse(throw new IllegalArgumentException("Cannot resolve CIs"))).foldLeft(false) { (b, id) =>
      ctx.logOutput(s"Validating configuration item $id for illegal character")
      val invalidChar = findInvalid(id)
      if (invalidChar.isDefined) {
        ctx.logError(s"$id contains illegal character ${charToString(invalidChar.get)}")
      }
      invalidChar.isDefined || b
    }
  }

  private[steps] def findInvalidChar(invalidChars: String)(id: String): Option[Char] = {
    invalidChars.find(id.toCharArray.contains)
  }
}
