package com.xebialabs.deployit.provision
package resolver.placeholder

import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind}
import com.xebialabs.deployit.plugin.api.udm.{DeployedApplication, Template}
import com.xebialabs.deployit.plugin.api.validation.ValidationMessage._
import com.xebialabs.deployit.service.replacement.{ConsolidatedDictionary, MustachePlaceholderScanner}

import scala.jdk.CollectionConverters._
import scala.util.{Failure, Try}

class MissingTemplatePlaceholderCollector(consolidatedDictionary: ConsolidatedDictionary) {
  private val stringPropertyTypes = Seq(PropertyKind.STRING, PropertyKind.SET_OF_STRING, PropertyKind.MAP_STRING_STRING, PropertyKind.LIST_OF_STRING)

  val scanner = new MustachePlaceholderScanner()

  def missingPlaceholders(deployedApplication: DeployedApplication, template: Template): Unit =
    template
      .getType
      .getDescriptor
      .getPropertyDescriptors
      .asScala
      .filter(pd => stringPropertyTypes.contains(pd.getKind))
      .foreach(pd => {
        val propertyValue = template.getProperty[Any](pd.getName)
        if (propertyValue.nonEmpty) {
          val missingValueCollector = new JHashSet[String]()
          Try(consolidatedDictionary.resolve(propertyValue, pd, missingValueCollector)) match {
            case _ if !missingValueCollector.isEmpty =>
              setValidationMessage(deployedApplication, template, pd, missingValueCollector.asScala.mkString("{{", "}}\", \"{{", "}}"))
            case Failure(_) =>
              setValidationMessage(deployedApplication, template, pd, propertyValue)
            case _ =>
          }
        }
      })

  private[this] def setValidationMessage(deployedApplication: DeployedApplication, template: Template, pd: PropertyDescriptor, value: Any): Boolean = {
    deployedApplication
      .get$validationMessages
      .add(warn(template.getId, pd.getName,
        s"""Could not resolve the property value(s) "$value" using the dictionary"""))
  }
}
