package ai.digital.deploy.task.steplog.queue

import java.util

import ai.digital.configuration.central.deploy.TaskerSystemProperties
import ai.digital.deploy.task.steplog.TaskStepLogStore
import com.xebialabs.deployit.core.events.{TaskStepLogDeleteEvent, TaskStepLogEvent}
import com.xebialabs.xldeploy.jms.adapter.TaskStepLogQueueNameResolver
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.context.ApplicationEventPublisher
import org.springframework.jms.core.JmsTemplate
import org.springframework.scheduling.annotation.Scheduled
import org.springframework.stereotype.Component

import scala.jdk.CollectionConverters._

case class TaskStepLogEventKey(taskId: String, stepPath: String, lineNumber: Integer, log: String, attempt: Integer) {
  def this(event: TaskStepLogEvent) = this(event.taskId, event.stepPath, event.lineNumber, event.log, event.attempt)
}

@ConditionalOnProperty(prefix = "deploy.task", name = Array("resilient"), havingValue = "true", matchIfMissing = false)
@Component
@Autowired
class TaskStepLogProducer(@Qualifier("taskStepLogJmsTemplate") jmsTemplate: JmsTemplate,
                          taskStepLogQueueNameResolver: TaskStepLogQueueNameResolver,
                          @Autowired publisher: ApplicationEventPublisher,
                          taskerSystemProperties: TaskerSystemProperties) extends TaskStepLogStore with Logging {

  private val taskStepLogEvents = new util.LinkedHashMap[TaskStepLogEventKey, TaskStepLogEvent]()

  private def enabled: Boolean = taskerSystemProperties.resilient

  private def flushThreshold: Int = taskerSystemProperties.events.taskStepLog.queue.eventFlushThreshold

  override def sendStepLogApplicationEvent(event: TaskStepLogEvent): Unit = {
    publisher.publishEvent(event)
  }

  override def sendStepLogEvent(event: TaskStepLogEvent): Unit =
    if (enabled) {
      taskStepLogEvents.synchronized {
        taskStepLogEvents.put(new TaskStepLogEventKey(event), event)
        if (taskStepLogEvents.size() >= flushThreshold) {
          logger.debug(s"Flushing change set events ${taskStepLogEvents.size()}")
          flushChangeSetEvents()
        }
      }
    }

  override def sendStepLogDeleteEvent(event: TaskStepLogDeleteEvent): Unit =
    if (enabled) {
      taskStepLogEvents.synchronized {
        val entries = taskStepLogEvents.entrySet().iterator()
        while (entries.hasNext) {
          val entry = entries.next()
          if (event.taskId == entry.getValue.taskId)
            entries.remove()
        }
      }
      jmsTemplate.convertAndSend(taskStepLogQueueNameResolver.getTaskStepLogQueueName, event)
    }

  @Scheduled(fixedDelayString = "#{@taskStepLogQueueFlushResolver.getEventFlushPeriodInMillis()}")
  def sendPeriodicallyChangeSetEvents(): Unit = {
    if (!taskStepLogEvents.isEmpty) {
      taskStepLogEvents.synchronized {
        logger.debug(s"Flushing change set events ${taskStepLogEvents.size()}")
        flushChangeSetEvents()
      }
    }
  }

  private def flushChangeSetEvents(): Unit = {
    val entries = taskStepLogEvents.entrySet()
    val messageEntries = entries.asScala
      .map(_.getValue)
      .toSeq
    jmsTemplate.convertAndSend(taskStepLogQueueNameResolver.getTaskStepLogQueueName, TaskStepLogEventMessage(messageEntries))
    entries.clear()
  }
}

@ConditionalOnProperty(prefix = "deploy.task", name = Array("resilient"), havingValue = "false", matchIfMissing = true)
@Component
class DummyTaskStepLogProducer extends TaskStepLogStore {
  override def sendStepLogEvent(event: TaskStepLogEvent): Unit = {}

  override def sendStepLogDeleteEvent(event: TaskStepLogDeleteEvent): Unit = {}

  override def sendStepLogApplicationEvent(event: TaskStepLogEvent): Unit = {}
}

