package ai.digital.deploy.task.status.queue

import ai.digital.deploy.engine.tasker.service.TaskPathStatusService
import com.xebialabs.deployit.core.events.{TaskPathStatusDeleteEvent, TaskPathStatusEvent}
import grizzled.slf4j.Logging
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty
import org.springframework.jms.annotation.JmsListener
import org.springframework.jms.support.converter.MessageConverter
import org.springframework.stereotype.Component

import javax.jms.TextMessage

@ConditionalOnProperty(prefix = "deploy.task", name = Array("resilient"), havingValue = "true", matchIfMissing = false)
@Component
class TaskPathStatusConsumer(taskPathStatusService: TaskPathStatusService,
                             jacksonJmsMessageConverter: MessageConverter) extends Logging {

  @JmsListener(destination = "#{@taskPathStatusQueueNameResolver.getTaskPathStatusQueueName()}", containerFactory = "xlJmsListenerContainerFactory")
  def receiveTextMessage(message: TextMessage): Unit =
    jacksonJmsMessageConverter.fromMessage(message) match {
      case taskPathStatusEventMessage: TaskPathStatusEventMessage =>
        logger.debug(
          s"Received taskPathStatusEvent ${message.getJMSMessageID} timestamp: ${message.getJMSTimestamp} size: ${taskPathStatusEventMessage.events.size}"
        )

        taskPathStatusService.batchCreate(taskPathStatusEventMessage.events, message.getJMSTimestamp)

      case taskPathStatusEvent: TaskPathStatusEvent =>
        logger.debug(
          s"Received taskPathStatusEvent ${message.getJMSMessageID} timestamp: ${message.getJMSTimestamp} taskId: ${taskPathStatusEvent.taskId} " +
            s"path: ${taskPathStatusEvent.path} status: ${taskPathStatusEvent.status}"
        )

        taskPathStatusService.create(taskPathStatusEvent.taskId, taskPathStatusEvent.path, taskPathStatusEvent.status, message.getJMSTimestamp)
      case taskPathStatusDeleteEvent: TaskPathStatusDeleteEvent =>
        logger.debug(
          s"Received taskPathStatusDeleteEvent ${message.getJMSMessageID} timestamp: ${message.getJMSTimestamp} taskId: ${taskPathStatusDeleteEvent.taskId}"
        )

        taskPathStatusService.delete(taskPathStatusDeleteEvent.taskId)
      case _ =>
        logger.error(s"Received unknown ${message.getJMSMessageID} timestamp: ${message.getJMSTimestamp} [$message]")
    }
}
