/**
 * Copyright 2014-2019 XebiaLabs Inc. and its affiliates. Use is subject to terms of the enclosed Legal Notice.
 */

package com.xebialabs.deployit.plugin.stitch.service.engine.context;

import com.fasterxml.jackson.databind.JsonNode;
import com.xebialabs.xlplatform.documentation.PublicApi;

import java.util.List;

/**
 * InvocationDocument is abstraction for original input document for the executed Stitch Invocation.
 * It is used for extracting single or list of values based on pathExpressions.
 * Actual sub-class is {@link JsonInputContext}.
 * JsonInputContext is used for Json/YAML inputs. If YAML was in original input it is transformed into
 * equivalent Json document.
 * Extraction of values and lists is done by calling methods with valid JsonPath expression
 * (see referral implementation of jayway.jsonPath:json-path)
 */
@PublicApi
public interface InputContext {
  /**
  * Returns true if path exists and has assigned value to it.
  * @param pathExpression on which we run check
  * @return evaluated flag for given {@code pathExpression}
  */
  boolean pathExistsWithValue(String pathExpression);

  /**
   * Returns {@link String} value in given path. If path doesn't exist or isn't single value {@code null} will be returned.
   * @param pathExpression on which we run check
   * @return evaluated {@link String} value for given {@code pathExpression}
   */
  String getStringValue(String pathExpression);

  /**
   * Returns {@link Boolean} value in given path. If path doesn't exist or isn't single value {@code null} will be returned.
   * @param pathExpression on which we run check
   * @return evaluated @link Boolean} value for given {@code pathExpression}
   */
  Boolean getBooleanValue(String pathExpression);

  /**
   * Returns {@link Integer} value in given path. If path doesn't exist or isn't single value {@code null} will be returned.
   * @param pathExpression on which we run check
   * @return evaluated {@link Integer} value for given {@code pathExpression}
   */
  Integer getIntegerValue(String pathExpression);

  /**
   * Returns {@link Long} value in given path. If path doesn't exist or isn't single value {@code null} will be returned.
   * @param pathExpression on which we run check
   * @return evaluated {@link Long} value for given {@code pathExpression}
   */
  Long getLongValue(String pathExpression);

  /**
   * Returns {@link Double} value in given path. If path doesn't exist or isn't single value {@code null} will be returned.
   * @param pathExpression on which we run check
   * @return evaluated {@link Double} value for given {@code pathExpression}
   */
  Double getDoubleValue(String pathExpression);

  /**
   * Returns {@link JsonNode} value in given path. If path doesn't exist {@code null} will be returned.
   * @param pathExpression on which we run check
   * @return evaluated {@link JsonNode} value for given {@code pathExpression}
   * @see JsonNode
   */
  JsonNode getJsonNode(String pathExpression);

  /**
   * Returns {@link List} for given path. If path doesn't exist {@code null} will be returned.
   * @param pathExpression on which we run check
   * @return evaluated {@link List} value for given {@code pathExpression}
   */
  List<String> getStringList(String pathExpression);

  /**
   * Returns {@link List} of {@link Boolean} for given path. If path doesn't exist {@code null} will be returned.
   * @param pathExpression on which we run check
   * @return evaluated {@link List} of {@link Boolean} value for given {@code pathExpression}
   */
  List<Boolean> getBooleanList(String pathExpression);

  /**
   * Returns {@link List} of {@link Integer} for given path. If path doesn't exist {@code null} will be returned.
   * @param pathExpression on which we run check
   * @return evaluated {@link List} of {@link Integer} value for given {@code pathExpression}
   */
  List<Integer> getIntegerList(String pathExpression);

  /**
   * Returns {@link List} of {@link Long} for given path. If path doesn't exist {@code null} will be returned.
   * @param pathExpression on which we run check
   * @return evaluated {@link List} of {@link Long} value for given {@code pathExpression}
   */
  List<Long> getLongList(String pathExpression);

  /**
   * Returns {@link List} of {@link Double} for given path. If path doesn't exist {@code null} will be returned.
   * @param pathExpression on which we run check
   * @return evaluated {@link List} of {@link Double} value for given {@code pathExpression}
   */
  List<Double> getDoubleList(String pathExpression);

  /**
   * Returns {@link List} of {@link JsonNode} for given path. If path doesn't exist {@code null} will be returned.
   * @param pathExpression on which we run check
   * @return evaluated {@link List} of {@link JsonNode} value for given {@code pathExpression}
   * @see JsonNode
   */
  List<JsonNode> getJsonNodeList(String pathExpression);
}
