/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.xds;

import com.google.common.base.MoreObjects;
import com.google.common.base.Preconditions;
import com.google.common.collect.Sets;
import io.grpc.Attributes;
import io.grpc.ConnectivityState;
import io.grpc.ConnectivityStateInfo;
import io.grpc.EquivalentAddressGroup;
import io.grpc.InternalLogId;
import io.grpc.LoadBalancer;
import io.grpc.Status;
import io.grpc.SynchronizationContext;
import io.grpc.xds.InternalXdsAttributes;
import io.grpc.xds.XdsLogger;
import io.grpc.xds.XdsNameResolver;
import io.grpc.xds.XdsSubchannelPickers;
import io.grpc.xds.XxHash64;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Random;
import javax.annotation.Nullable;

final class RingHashLoadBalancer
extends LoadBalancer {
    private static final Attributes.Key<Ref<ConnectivityStateInfo>> STATE_INFO = Attributes.Key.create((String)"state-info");
    private static final Status RPC_HASH_NOT_FOUND = Status.INTERNAL.withDescription("RPC hash not found. Probably a bug because xds resolver config selector always generates a hash.");
    private static final XxHash64 hashFunc = XxHash64.INSTANCE;
    private final XdsLogger logger;
    private final SynchronizationContext syncContext;
    private final Map<EquivalentAddressGroup, LoadBalancer.Subchannel> subchannels = new HashMap<EquivalentAddressGroup, LoadBalancer.Subchannel>();
    private final LoadBalancer.Helper helper;
    private List<RingEntry> ring;
    private ConnectivityState currentState;
    private Iterator<LoadBalancer.Subchannel> connectionAttemptIterator = this.subchannels.values().iterator();
    private final Random random = new Random();

    RingHashLoadBalancer(LoadBalancer.Helper helper) {
        this.helper = (LoadBalancer.Helper)Preconditions.checkNotNull((Object)helper, (Object)"helper");
        this.syncContext = (SynchronizationContext)Preconditions.checkNotNull((Object)helper.getSynchronizationContext(), (Object)"syncContext");
        this.logger = XdsLogger.withLogId(InternalLogId.allocate((String)"ring_hash_lb", (String)helper.getAuthority()));
        this.logger.log(XdsLogger.XdsLogLevel.INFO, "Created");
    }

    public void handleResolvedAddresses(LoadBalancer.ResolvedAddresses resolvedAddresses) {
        this.logger.log(XdsLogger.XdsLogLevel.DEBUG, "Received resolution result: {0}", resolvedAddresses);
        List addrList = resolvedAddresses.getAddresses();
        if (addrList.isEmpty()) {
            this.handleNameResolutionError(Status.UNAVAILABLE.withDescription("Ring hash lb error: EDS resolution was successful, but returned server addresses are empty."));
            return;
        }
        Map<EquivalentAddressGroup, EquivalentAddressGroup> latestAddrs = RingHashLoadBalancer.stripAttrs(addrList);
        HashSet removedAddrs = Sets.newHashSet((Iterable)Sets.difference(this.subchannels.keySet(), latestAddrs.keySet()));
        RingHashConfig config = (RingHashConfig)resolvedAddresses.getLoadBalancingPolicyConfig();
        HashMap<EquivalentAddressGroup, Long> serverWeights = new HashMap<EquivalentAddressGroup, Long>();
        long totalWeight = 0L;
        for (EquivalentAddressGroup eag : addrList) {
            Long weight = (Long)eag.getAttributes().get(InternalXdsAttributes.ATTR_SERVER_WEIGHT);
            if (weight == null) {
                weight = 1L;
            }
            totalWeight += weight.longValue();
            EquivalentAddressGroup addrKey = RingHashLoadBalancer.stripAttrs(eag);
            if (serverWeights.containsKey(addrKey)) {
                serverWeights.put(addrKey, (Long)serverWeights.get(addrKey) + weight);
            } else {
                serverWeights.put(addrKey, weight);
            }
            LoadBalancer.Subchannel existingSubchannel = this.subchannels.get(addrKey);
            if (existingSubchannel != null) {
                existingSubchannel.updateAddresses(Collections.singletonList(eag));
                continue;
            }
            Attributes attr = Attributes.newBuilder().set(STATE_INFO, new Ref<ConnectivityStateInfo>(ConnectivityStateInfo.forNonError((ConnectivityState)ConnectivityState.IDLE))).build();
            final LoadBalancer.Subchannel subchannel = this.helper.createSubchannel(LoadBalancer.CreateSubchannelArgs.newBuilder().setAddresses(eag).setAttributes(attr).build());
            subchannel.start(new LoadBalancer.SubchannelStateListener(){

                public void onSubchannelState(ConnectivityStateInfo newState) {
                    RingHashLoadBalancer.this.processSubchannelState(subchannel, newState);
                }
            });
            this.subchannels.put(addrKey, subchannel);
        }
        long minWeight = (Long)Collections.min(serverWeights.values());
        double normalizedMinWeight = (double)minWeight / (double)totalWeight;
        double scale = Math.min(Math.ceil(normalizedMinWeight * (double)config.minRingSize) / normalizedMinWeight, (double)config.maxRingSize);
        this.ring = RingHashLoadBalancer.buildRing(serverWeights, totalWeight, scale);
        ArrayList<LoadBalancer.Subchannel> removedSubchannels = new ArrayList<LoadBalancer.Subchannel>();
        for (EquivalentAddressGroup addr : removedAddrs) {
            removedSubchannels.add(this.subchannels.remove(addr));
        }
        this.connectionAttemptIterator = this.subchannels.values().iterator();
        int randomAdvance = this.random.nextInt(this.subchannels.size());
        while (randomAdvance-- > 0) {
            this.connectionAttemptIterator.next();
        }
        this.updateBalancingState();
        for (LoadBalancer.Subchannel subchann : removedSubchannels) {
            RingHashLoadBalancer.shutdownSubchannel(subchann);
        }
    }

    private static List<RingEntry> buildRing(Map<EquivalentAddressGroup, Long> serverWeights, long totalWeight, double scale) {
        ArrayList<RingEntry> ring = new ArrayList<RingEntry>();
        double currentHashes = 0.0;
        double targetHashes = 0.0;
        for (Map.Entry<EquivalentAddressGroup, Long> entry : serverWeights.entrySet()) {
            EquivalentAddressGroup addrKey = entry.getKey();
            double normalizedWeight = (double)entry.getValue().longValue() / (double)totalWeight;
            StringBuilder sb = new StringBuilder(addrKey.getAddresses().toString());
            sb.append('_');
            int lengthWithoutCounter = sb.length();
            targetHashes += scale * normalizedWeight;
            long i = 0L;
            while (currentHashes < targetHashes) {
                sb.append(i);
                long hash = hashFunc.hashAsciiString(sb.toString());
                ring.add(new RingEntry(hash, addrKey));
                ++i;
                currentHashes += 1.0;
                sb.setLength(lengthWithoutCounter);
            }
        }
        Collections.sort(ring);
        return Collections.unmodifiableList(ring);
    }

    public void handleNameResolutionError(Status error) {
        if (this.currentState != ConnectivityState.READY) {
            this.helper.updateBalancingState(ConnectivityState.TRANSIENT_FAILURE, (LoadBalancer.SubchannelPicker)new XdsSubchannelPickers.ErrorPicker(error));
        }
    }

    public void shutdown() {
        this.logger.log(XdsLogger.XdsLogLevel.INFO, "Shutdown");
        for (LoadBalancer.Subchannel subchannel : this.subchannels.values()) {
            RingHashLoadBalancer.shutdownSubchannel(subchannel);
        }
        this.subchannels.clear();
    }

    private void updateBalancingState() {
        ConnectivityState overallState;
        Preconditions.checkState((!this.subchannels.isEmpty() ? 1 : 0) != 0, (Object)"no subchannel has been created");
        boolean startConnectionAttempt = false;
        int numIdle = 0;
        int numReady = 0;
        int numConnecting = 0;
        int numTransientFailure = 0;
        for (LoadBalancer.Subchannel subchannel : this.subchannels.values()) {
            ConnectivityState state = ((ConnectivityStateInfo)RingHashLoadBalancer.getSubchannelStateInfoRef((LoadBalancer.Subchannel)subchannel).value).getState();
            if (state == ConnectivityState.READY) {
                ++numReady;
                break;
            }
            if (state == ConnectivityState.TRANSIENT_FAILURE) {
                ++numTransientFailure;
                continue;
            }
            if (state == ConnectivityState.CONNECTING) {
                ++numConnecting;
                continue;
            }
            if (state != ConnectivityState.IDLE) continue;
            ++numIdle;
        }
        if (numReady > 0) {
            overallState = ConnectivityState.READY;
        } else if (numTransientFailure >= 2) {
            overallState = ConnectivityState.TRANSIENT_FAILURE;
            startConnectionAttempt = numConnecting == 0;
        } else if (numConnecting > 0) {
            overallState = ConnectivityState.CONNECTING;
        } else if (numTransientFailure == 1 && this.subchannels.size() > 1) {
            overallState = ConnectivityState.CONNECTING;
            startConnectionAttempt = true;
        } else if (numIdle > 0) {
            overallState = ConnectivityState.IDLE;
        } else {
            overallState = ConnectivityState.TRANSIENT_FAILURE;
            startConnectionAttempt = true;
        }
        RingHashPicker picker = new RingHashPicker(this.syncContext, this.ring, this.subchannels);
        this.helper.updateBalancingState(overallState, (LoadBalancer.SubchannelPicker)picker);
        this.currentState = overallState;
        if (startConnectionAttempt) {
            if (!this.connectionAttemptIterator.hasNext()) {
                this.connectionAttemptIterator = this.subchannels.values().iterator();
            }
            this.connectionAttemptIterator.next().requestConnection();
        }
    }

    private void processSubchannelState(LoadBalancer.Subchannel subchannel, ConnectivityStateInfo stateInfo) {
        if (this.subchannels.get(RingHashLoadBalancer.stripAttrs(subchannel.getAddresses())) != subchannel) {
            return;
        }
        if (stateInfo.getState() == ConnectivityState.TRANSIENT_FAILURE || stateInfo.getState() == ConnectivityState.IDLE) {
            this.helper.refreshNameResolution();
        }
        this.updateConnectivityState(subchannel, stateInfo);
        this.updateBalancingState();
    }

    private void updateConnectivityState(LoadBalancer.Subchannel subchannel, ConnectivityStateInfo stateInfo) {
        Ref<ConnectivityStateInfo> subchannelStateRef = RingHashLoadBalancer.getSubchannelStateInfoRef(subchannel);
        ConnectivityState previousConnectivityState = ((ConnectivityStateInfo)subchannelStateRef.value).getState();
        if (previousConnectivityState == ConnectivityState.TRANSIENT_FAILURE && (stateInfo.getState() == ConnectivityState.CONNECTING || stateInfo.getState() == ConnectivityState.IDLE)) {
            return;
        }
        subchannelStateRef.value = stateInfo;
    }

    private static void shutdownSubchannel(LoadBalancer.Subchannel subchannel) {
        subchannel.shutdown();
        RingHashLoadBalancer.getSubchannelStateInfoRef((LoadBalancer.Subchannel)subchannel).value = ConnectivityStateInfo.forNonError((ConnectivityState)ConnectivityState.SHUTDOWN);
    }

    private static Map<EquivalentAddressGroup, EquivalentAddressGroup> stripAttrs(List<EquivalentAddressGroup> groupList) {
        HashMap<EquivalentAddressGroup, EquivalentAddressGroup> addrs = new HashMap<EquivalentAddressGroup, EquivalentAddressGroup>(groupList.size() * 2);
        for (EquivalentAddressGroup group : groupList) {
            addrs.put(RingHashLoadBalancer.stripAttrs(group), group);
        }
        return addrs;
    }

    private static EquivalentAddressGroup stripAttrs(EquivalentAddressGroup eag) {
        return new EquivalentAddressGroup(eag.getAddresses());
    }

    private static Ref<ConnectivityStateInfo> getSubchannelStateInfoRef(LoadBalancer.Subchannel subchannel) {
        return (Ref)Preconditions.checkNotNull((Object)((Ref)subchannel.getAttributes().get(STATE_INFO)), (Object)"STATE_INFO");
    }

    static final class RingHashConfig {
        final long minRingSize;
        final long maxRingSize;

        RingHashConfig(long minRingSize, long maxRingSize) {
            Preconditions.checkArgument((minRingSize > 0L ? 1 : 0) != 0, (Object)"minRingSize <= 0");
            Preconditions.checkArgument((maxRingSize > 0L ? 1 : 0) != 0, (Object)"maxRingSize <= 0");
            Preconditions.checkArgument((minRingSize <= maxRingSize ? 1 : 0) != 0, (Object)"minRingSize > maxRingSize");
            this.minRingSize = minRingSize;
            this.maxRingSize = maxRingSize;
        }

        public String toString() {
            return MoreObjects.toStringHelper((Object)this).add("minRingSize", this.minRingSize).add("maxRingSize", this.maxRingSize).toString();
        }
    }

    private static final class Ref<T> {
        T value;

        Ref(T value) {
            this.value = value;
        }
    }

    private static final class RingEntry
    implements Comparable<RingEntry> {
        private final long hash;
        private final EquivalentAddressGroup addrKey;

        private RingEntry(long hash, EquivalentAddressGroup addrKey) {
            this.hash = hash;
            this.addrKey = addrKey;
        }

        @Override
        public int compareTo(RingEntry entry) {
            return Long.compare(this.hash, entry.hash);
        }
    }

    private static final class SubchannelView {
        private final LoadBalancer.Subchannel subchannel;
        private final ConnectivityStateInfo stateInfo;

        private SubchannelView(LoadBalancer.Subchannel subchannel, ConnectivityStateInfo stateInfo) {
            this.subchannel = subchannel;
            this.stateInfo = stateInfo;
        }
    }

    private static final class RingHashPicker
    extends LoadBalancer.SubchannelPicker {
        private final SynchronizationContext syncContext;
        private final List<RingEntry> ring;
        private final Map<EquivalentAddressGroup, SubchannelView> pickableSubchannels;

        private RingHashPicker(SynchronizationContext syncContext, List<RingEntry> ring, Map<EquivalentAddressGroup, LoadBalancer.Subchannel> subchannels) {
            this.syncContext = syncContext;
            this.ring = ring;
            this.pickableSubchannels = new HashMap<EquivalentAddressGroup, SubchannelView>(subchannels.size());
            for (Map.Entry<EquivalentAddressGroup, LoadBalancer.Subchannel> entry : subchannels.entrySet()) {
                LoadBalancer.Subchannel subchannel = entry.getValue();
                ConnectivityStateInfo stateInfo = (ConnectivityStateInfo)((Ref)subchannel.getAttributes().get((Attributes.Key)STATE_INFO)).value;
                this.pickableSubchannels.put(entry.getKey(), new SubchannelView(subchannel, stateInfo));
            }
        }

        public LoadBalancer.PickResult pickSubchannel(LoadBalancer.PickSubchannelArgs args) {
            int mid;
            block8: {
                Long requestHash = (Long)args.getCallOptions().getOption(XdsNameResolver.RPC_HASH_KEY);
                if (requestHash == null) {
                    return LoadBalancer.PickResult.withError((Status)RPC_HASH_NOT_FOUND);
                }
                int low = 0;
                int high = this.ring.size();
                do {
                    long midValL;
                    if ((mid = (low + high) / 2) == this.ring.size()) {
                        mid = 0;
                        break block8;
                    }
                    long midVal = this.ring.get(mid).hash;
                    long l = midValL = mid == 0 ? 0L : this.ring.get(mid - 1).hash;
                    if (requestHash <= midVal && requestHash > midValL) break block8;
                    if (midVal < requestHash) {
                        low = mid + 1;
                        continue;
                    }
                    high = mid - 1;
                } while (low <= high);
                mid = 0;
            }
            boolean foundFirstNonFailed = false;
            LoadBalancer.Subchannel firstSubchannel = null;
            LoadBalancer.Subchannel secondSubchannel = null;
            for (int i = 0; i < this.ring.size(); ++i) {
                LoadBalancer.PickResult maybeBuffer;
                int index = (mid + i) % this.ring.size();
                EquivalentAddressGroup addrKey = this.ring.get(index).addrKey;
                SubchannelView subchannel = this.pickableSubchannels.get(addrKey);
                if (subchannel.stateInfo.getState() == ConnectivityState.READY) {
                    return LoadBalancer.PickResult.withSubchannel((LoadBalancer.Subchannel)subchannel.subchannel);
                }
                if (firstSubchannel == null) {
                    firstSubchannel = subchannel.subchannel;
                    maybeBuffer = this.pickSubchannelsNonReady(subchannel);
                    if (maybeBuffer == null) continue;
                    return maybeBuffer;
                }
                if (subchannel.subchannel != firstSubchannel && secondSubchannel == null) {
                    secondSubchannel = subchannel.subchannel;
                    maybeBuffer = this.pickSubchannelsNonReady(subchannel);
                    if (maybeBuffer == null) continue;
                    return maybeBuffer;
                }
                if (subchannel.subchannel == firstSubchannel || subchannel.subchannel == secondSubchannel || foundFirstNonFailed) continue;
                this.pickSubchannelsNonReady(subchannel);
                if (subchannel.stateInfo.getState() == ConnectivityState.TRANSIENT_FAILURE) continue;
                foundFirstNonFailed = true;
            }
            SubchannelView originalSubchannel = this.pickableSubchannels.get(this.ring.get(mid).addrKey);
            return LoadBalancer.PickResult.withError((Status)originalSubchannel.stateInfo.getStatus());
        }

        @Nullable
        private LoadBalancer.PickResult pickSubchannelsNonReady(SubchannelView subchannel) {
            if (subchannel.stateInfo.getState() == ConnectivityState.TRANSIENT_FAILURE || subchannel.stateInfo.getState() == ConnectivityState.IDLE) {
                final LoadBalancer.Subchannel finalSubchannel = subchannel.subchannel;
                this.syncContext.execute(new Runnable(){

                    @Override
                    public void run() {
                        finalSubchannel.requestConnection();
                    }
                });
            }
            if (subchannel.stateInfo.getState() == ConnectivityState.CONNECTING || subchannel.stateInfo.getState() == ConnectivityState.IDLE) {
                return LoadBalancer.PickResult.withNoResult();
            }
            return null;
        }
    }
}

