package com.xebialabs.deployit.engine.tasker.repository.sql

import java.io.ByteArrayInputStream
import java.sql.{PreparedStatement, ResultSet}

import com.xebialabs.deployit.core.sql._
import com.xebialabs.deployit.core.sql.lock.{LockNameRegistry, SqlLockRepository}
import com.xebialabs.deployit.core.sql.spring.Setter.setString
import com.xebialabs.deployit.engine.api.distribution.TaskExecutionWorkerRepository
import com.xebialabs.deployit.engine.api.dto.Worker
import com.xebialabs.xlplatform.utils.ResourceManagement
import grizzled.slf4j.Logging
import org.apache.commons.io.IOUtils
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.Transactional

import scala.jdk.CollectionConverters._

@Component
@Transactional("mainTransactionManager")
class SqlWorkerRepository(@Autowired @Qualifier("mainJdbcTemplate") override val jdbcTemplate: JdbcTemplate)
                         (@Autowired @Qualifier("mainSchema") override implicit val schemaInfo: SchemaInfo)
  extends TaskExecutionWorkerRepository with SqlLockRepository with  WorkersQueries with Logging {

  override def storeWorker(worker: Worker): Unit = {
    val (publicKeyStream, publicKeyLength) = if (worker.publicKey != null)
      (new ByteArrayInputStream(worker.publicKey), worker.publicKey.length)
    else (null, 0)
    executeWithLock(LockNameRegistry.XLD_WORKERS_LOCK) {
      val result = jdbcTemplate.update(UPDATE_WORKER, (ps: PreparedStatement) => {
        ps.setBinaryStream(1, publicKeyStream, publicKeyLength)
        setString(ps, 2, worker.configurationHash)
        setString(ps,3, worker.name)
        setString(ps,4, worker.address)
      })
      if (result == 0) {
        jdbcTemplate.update(INSERT_WORKER, (ps: PreparedStatement) => {
          setString(ps,1, worker.address)
          ps.setBinaryStream(2, Option(worker.publicKey).map(new ByteArrayInputStream(_)).orNull, publicKeyLength)
          setString(ps,3, worker.configurationHash)
          setString(ps,4, worker.name)
        })
      }
    }
  }

  val constructWorker: RowMapper[Worker] = (rs: ResultSet, _: Int) => {
    val id = rs.getInt(WorkersSchema.worker_id.name).asInstanceOf[Integer]
    val address = rs.getString(WorkersSchema.address.name)
    val publicKeyStream = rs.getBinaryStream(WorkersSchema.public_key.name)
    val publicKey = if (publicKeyStream != null) ResourceManagement.using(publicKeyStream) {
      IOUtils.toByteArray
    } else null
    val hash = rs.getString(WorkersSchema.configuration_hash.name)
    val workerName = rs.getString(WorkersSchema.worker_name.name)
    Worker(id, workerName, address, publicKey, hash)
  }

  override def getWorkerByAddress(address: String): Option[Worker] = {
    jdbcTemplate.query(SELECT_WORKER_BY_ADDRESS, constructWorker, address).asScala.headOption
  }

  override def getWorker(id: Integer): Option[Worker] = {
    jdbcTemplate.query(SELECT_WORKER, constructWorker, id).asScala.headOption
  }

  override def removeWorker(address: String): Unit = {
    if (jdbcTemplate.queryForObject(COUNT_TASKS_OF_WORKER, classOf[Integer], address).intValue() == 0) {
      jdbcTemplate.update(DELETE_WORKER_BY_ADDRESS, address)
    } else {
      logger.debug(s"Cannot delete worker $address, since it is still running active tasks.")
    }
  }

  override def removeWorker(id: Integer): Unit = {
    jdbcTemplate.update(DELETE_WORKER, id)
  }

  override def listWorkers: List[(Worker, Integer, Integer)] = {
    jdbcTemplate.queryForList(SELECT_ALL_WORKERS).asScala.map { result =>
      val id = asInteger(result.get(WorkersSchema.worker_id.name))
      val address = result.get(WorkersSchema.address.name).asInstanceOf[String]
      val hash = result.get(WorkersSchema.configuration_hash.name).asInstanceOf[String]
      val deploymentTasks = asInteger(result.get(WorkersSchema.deployment_tasks.name))
      val controlTasks = asInteger(result.get(WorkersSchema.control_tasks.name))
      val workerName = result.get(WorkersSchema.worker_name.name).asInstanceOf[String]
      (Worker(id, workerName, address, null, hash), deploymentTasks, controlTasks)
    }.toList
  }

  override def removeTasks(id: Integer): Unit = {
    jdbcTemplate.update(DELETE_TASKS_BY_WORKER, id)
  }
}

object WorkersSchema {
  val tableName: TableName = TableName("XLD_WORKERS")

  val worker_id: ColumnName = ColumnName("ID")
  val worker_name: ColumnName = ColumnName("name")
  val address: ColumnName = ColumnName("address")
  val configuration_hash: ColumnName = ColumnName("configuration_hash")
  val public_key: ColumnName = ColumnName("public_key")
  val deployment_tasks: ColumnName = ColumnName("deployment_tasks")
  val control_tasks: ColumnName = ColumnName("control_tasks")
}

trait WorkersQueries extends Queries {
  import WorkersSchema._
  import com.xebialabs.deployit.sql.base.schema.{ActiveTaskMetadataSchema => atmd, ActiveTaskSchema => at}

  lazy val INSERT_WORKER: String = sqlb"insert into $tableName ($address, $public_key, $configuration_hash, $worker_name) values (?,?,?,?)"
  lazy val UPDATE_WORKER: String = sqlb"update $tableName set $public_key = ?, $configuration_hash = ?, $worker_name = ? where $address = ?"
  lazy val DELETE_WORKER: String = sqlb"delete from $tableName where $worker_id = ?"
  lazy val DELETE_WORKER_BY_ADDRESS: String = sqlb"delete from $tableName where $address = ?"
  lazy val DELETE_TASKS_BY_WORKER: String = {
    import com.xebialabs.deployit.sql.base.schema.{ActiveTaskSchema => ats}
    sqlb"delete from ${ats.tableName} where ${ats.worker_id} = ?"
  }
  lazy val COUNT_TASKS_OF_WORKER: String = {
    import com.xebialabs.deployit.sql.base.schema.{ActiveTaskSchema => ats}
    sqlb"select count(*) from ${ats.tableName} where ${ats.worker_id} = (select $worker_id from $tableName where $address = ?)"
  }
  lazy val SELECT_WORKER: String = sqlb"select * from $tableName where $worker_id = ?"
  lazy val SELECT_WORKER_BY_ADDRESS = sqlb"select * from $tableName where $address = ?"

  lazy val SELECT_ALL_WORKERS: String =
    sqlb"""
          |select max(w.$worker_id) "ID", max(w.$address) "address", max(w.$configuration_hash) "configuration_hash", max(w.$worker_name) "name",
          |count(CASE WHEN atmd.${atmd.metadata_value} is null or ${schemaInfo.sqlDialect.castToString(atmd.metadata_value.tableAlias("atmd"))} IN ('CONTROL', 'INSPECTION') THEN NULL ELSE 1 END) "deployment_tasks",
          |count(CASE WHEN ${schemaInfo.sqlDialect.castToString(atmd.metadata_value.tableAlias("atmd"))} IN ('CONTROL', 'INSPECTION') THEN 1 ELSE NULL END) "control_tasks"
          |from $tableName w left join ${at.tableName} xat on w.$worker_id = xat.${at.worker_id}
          |left join ${atmd.tableName} atmd on (atmd.${atmd.task_id} = xat.${at.task_id} and atmd.${atmd.metadata_key} = 'taskType') group by w.$worker_id
      """.stripMargin
}
