package com.xebialabs.deployit.engine.tasker.repository.sql

import com.xebialabs.deployit.core.sql._
import com.xebialabs.deployit.engine.api.dto.TaskPathStatus
import com.xebialabs.deployit.engine.tasker.TaskId
import com.xebialabs.deployit.engine.tasker.repository.TaskPathStatusRepository
import com.xebialabs.deployit.sql.base.schema.TaskPathStatusSchema._
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, ResultSetExtractor, RowMapper}
import org.springframework.stereotype.Component
import org.springframework.transaction.annotation.{Isolation, Transactional}

import java.sql.ResultSet
import scala.jdk.CollectionConverters._

@Component
class SqlTaskPathStatusRepository(@Autowired @Qualifier("mainJdbcTemplate") val jdbcTemplate: JdbcTemplate)
                                 (@Autowired @Qualifier("mainSchema") implicit val schemaInfo: SchemaInfo)
  extends TaskPathStatusRepository with TaskPathStatusQueries with Logging {

  @Transactional(value = "mainTransactionManager", readOnly = true)
  override def findOne(taskId: TaskId, path: String): Option[TaskPathStatus] = {
    val resultSet = jdbcTemplate.query(SELECT_TASK_STATUS_BY_PATH, mapToRow, taskId, path).asScala.headOption
    resultSet
  }

  @Transactional("mainTransactionManager")
  override def delete(taskId: TaskId): Unit =
    jdbcTemplate.update(DELETE_TASK_STATUS, taskId)

  @Transactional(value = "mainTransactionManager", isolation = Isolation.SERIALIZABLE)
  override def createOrUpdate(taskId: TaskId, path: String, status: String, updateTime: Long): Unit =
    findUpdateTime(taskId, path) match {
      case Some((oldStatus, oldTime)) if updateTime >= oldTime && status != oldStatus =>
        jdbcTemplate.update(UPDATE_TASK_STATUS, status, updateTime, taskId, path, updateTime)
      case None =>
        jdbcTemplate.update(INSERT_TASK_STATUS, taskId, path, status, updateTime)
      case Some((oldStatus, oldTime)) =>
        if (status == oldStatus)
          logger.debug(s"Omitting insert because there is no status change $oldStatus (updateTime: $oldTime->$updateTime, taskId: $taskId, path: $path)")
        else
          logger.debug(s"Omitting insert because new update time $updateTime is before from persisted $oldTime (status: $oldStatus->$status, taskId: $taskId, path: $path)")
    }

  @Transactional(value = "mainTransactionManager", readOnly = true)
  override def findAll(taskId: TaskId): List[TaskPathStatus] =
    jdbcTemplate.query(SELECT_TASK_STATUS, mapToList, taskId)

  @Transactional(value = "mainTransactionManager", readOnly = true)
  override def findUpdateTime(taskId: TaskId, path: String): Option[(String, Long)] =
   jdbcTemplate.query(SELECT_TASK_STATUS_UPDATE_TIME, mapToTuple, taskId, path).asScala.headOption

  private def mapToRow: RowMapper[TaskPathStatus] =
    (rs: ResultSet, _) => TaskPathStatus(rs.getString(taskId.name), rs.getString(path.name), rs.getString(status.name))

  private def mapToTuple: RowMapper[(String, Long)] =
    (rs: ResultSet, _) => (rs.getString(status.name), rs.getLong(updateTime.name))

  private def mapToList: ResultSetExtractor[List[TaskPathStatus]] = rs => {
    var taskPathStatusList = List[TaskPathStatus]()
    while (rs.next) {
      val row = TaskPathStatus(rs.getString(taskId.name), rs.getString(path.name), rs.getString(status.name))
      taskPathStatusList ::= row
    }
    taskPathStatusList
  }
}

private[sql] trait TaskPathStatusQueries extends Queries {

  private[sql] lazy val INSERT_TASK_STATUS: String = sqlb"insert into $tableName ($taskId, $path, $status, $updateTime) values (?,?,?,?)"
  private[sql] lazy val UPDATE_TASK_STATUS: String = sqlb"update $tableName set $status = ?, $updateTime = ? where $taskId = ? and $path = ? and $updateTime <= ?"
  private[sql] lazy val SELECT_TASK_STATUS_BY_PATH: String = sqlb"select $taskId, $path, $status from $tableName where $taskId = ? and $path = ?"
  private[sql] lazy val SELECT_TASK_STATUS_UPDATE_TIME: String = sqlb"select $status, $updateTime from $tableName where $taskId = ? and $path = ?"
  private[sql] lazy val DELETE_TASK_STATUS: String = sqlb"delete from $tableName where $taskId = ?"
  private[sql] lazy val SELECT_TASK_STATUS: String = sqlb"select $taskId, $path, $status from $tableName where $taskId = ?"

}


