package com.xebialabs.deployit.task.archive.sql

import ai.digital.deploy.permissions.client.ReferencedPermissionServiceClient
import com.xebialabs.deployit.core.sql.batch.{BatchCommand, BatchCommandWithArgs, BatchExecutorRepository}
import com.xebialabs.deployit.core.sql.spring.{ExtendedArgumentPreparedStatementSetter, Setter, toMap}
import com.xebialabs.deployit.core.sql.{SqlCondition => cond, SqlFunction => func, _}
import com.xebialabs.deployit.engine.api.execution.{FetchMode, TaskExecutionState}
import com.xebialabs.deployit.engine.tasker.TaskNotFoundException
import com.xebialabs.deployit.security.RoleService
import com.xebialabs.deployit.security.permission.{PermissionHelper, PlatformPermissions}
import com.xebialabs.deployit.task.ArchivedTaskSearchParameters
import com.xebialabs.deployit.task.archive._
import com.xebialabs.deployit.task.archive.sql.StandardDeviationBucketsComputeUtility.bins
import com.xebialabs.deployit.task.archive.sql.reports._
import com.xebialabs.deployit.task.archive.sql.schema.{ArchivedDeploymentTasks, ArchivedTasks, ArchivedTasksShared}
import grizzled.slf4j.Logging
import org.apache.commons.lang.exception.ExceptionUtils
import org.joda.time.DateTime
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.dao.EmptyResultDataAccessException
import org.springframework.jdbc.core._
import org.springframework.jdbc.datasource.DataSourceUtils
import org.springframework.jdbc.support.JdbcUtils
import org.springframework.stereotype.Repository
import org.springframework.transaction.PlatformTransactionManager
import org.springframework.transaction.annotation.Transactional
import org.springframework.transaction.support.DefaultTransactionDefinition

import java.lang.Boolean.TRUE
import java.sql._
import java.util
import java.util.{UUID, stream}
import javax.sql.DataSource
import org.springframework.lang.Nullable

import scala.jdk.CollectionConverters._

@Repository
@Transactional("reportingTransactionManager")
class SqlTaskArchiveRepository @Autowired()(@Qualifier("reportingDataSource") val dataSource: DataSource,
                                            @Qualifier("reportingSchema") implicit val schemaInfo: SchemaInfo,
                                            @Qualifier("reportingBatchExecutorRepository") batchExecutorRepository: BatchExecutorRepository) extends BatchTaskArchiveRepository with SqlTaskArchiveQueries
  with Logging {

  private[this] val template = new JdbcTemplateWithStreamingQuery(dataSource)

  override def store(taskMessage: TaskMessage): Unit =
    try
      TaskArchiver(taskMessage).store(template)
    catch {
      case e: Exception =>
        error(s"Error storing archive message for taskId: ${taskMessage.taskId}: ${e.getMessage}")
        throw e
    }



  override def updateSecureCi(securedCi: Integer, environmentIds: Option[List[Integer]],
                              applicationIds: Option[List[Integer]], otherCiIds: Option[List[Integer]]): Unit = {
    try {
      environmentIds.map(envIds => template.batchUpdate(UPDATE_ENVIRONMENT_SECURE_CI, getSecureCiSetter(securedCi, envIds)))
      applicationIds.map(appIds => template.batchUpdate(UPDATE_APPLICATION_SECURE_CI, getSecureCiSetter(securedCi, appIds)))
      otherCiIds.map(otherIds => template.batchUpdate(UPDATE_CONTROL_TASK_SECURE_CI, getSecureCiSetter(securedCi, otherIds)))
    }
    catch {
      case e: Exception =>
        error(s"Error storing securedCi [$securedCi] batch: ${e.getMessage}")
        throw e
    }
  }

  override def updateSecureCi(ciId: Integer, securedCi: Integer): Unit = {
    try {
      template.update(UPDATE_ENVIRONMENT_SECURE_CI, securedCi, ciId)
      template.update(UPDATE_APPLICATION_SECURE_CI, securedCi, ciId)
      template.update(UPDATE_CONTROL_TASK_SECURE_CI, securedCi, ciId)
    }
    catch {
      case e: Exception =>
        error(s"Error storing securedCi [$securedCi]: ${e.getMessage}")
        throw e
    }
  }

  override def batchUpdateSecureCi(messages: BatchSecureCiMessage): Unit = {
    val envCommands = messages.environmentBatchMessages.map(envMessage =>
      BatchCommand(UPDATE_ENVIRONMENT_SECURE_CI, envMessage.securedCi, envMessage.securedCi)
    )
    val appCommands = messages.applicationBatchMessages.map(appMessage =>
      BatchCommand(UPDATE_APPLICATION_SECURE_CI, appMessage.securedCi, appMessage.securedCi)
    )
    val controlTaskCommands = messages.controlTaskBatchMessages.map(controlTaskMessage =>
      BatchCommand(UPDATE_CONTROL_TASK_SECURE_CI, controlTaskMessage.securedCi, controlTaskMessage.securedCi)
    )

    batchExecute(envCommands ++ appCommands ++ controlTaskCommands)
  }

  override def batchExecute(commands: Iterable[BatchCommandWithArgs]): Unit =
    batchExecutorRepository.execute(commands)

  override def updateSecuredDirectoryRef(securedDirectoryRef: String, environmentIds: Option[List[Integer]], applicationIds: Option[List[Integer]], otherCiIds: Option[List[Integer]]): Unit =
    try {
      environmentIds.map(envIds => template.batchUpdate(UPDATE_ENVIRONMENT_SECURED_DIRECTORY_REF, getSecuredDirectoryReferenceSetter(securedDirectoryRef, envIds)))
      applicationIds.map(appIds => template.batchUpdate(UPDATE_APPLICATION_SECURED_DIRECTORY_REF, getSecuredDirectoryReferenceSetter(securedDirectoryRef, appIds)))
      otherCiIds.map(otherIds => template.batchUpdate(UPDATE_CONTROL_TASK_SECURED_DIRECTORY_REF, getSecuredDirectoryReferenceSetter(securedDirectoryRef, otherIds)))
    }
    catch {
      case e: Exception =>
        error(s"Error storing securedDirectoryReference [$securedDirectoryRef] batch: ${e.getMessage}")
        throw e
    }

  override def updateSecuredDirectoryRef(ciId: Integer, securedDirectoryRef: String): Unit =
    try {
      template.update(UPDATE_ENVIRONMENT_SECURED_DIRECTORY_REF, securedDirectoryRef, ciId)
      template.update(UPDATE_APPLICATION_SECURED_DIRECTORY_REF, securedDirectoryRef, ciId)
      template.update(UPDATE_CONTROL_TASK_SECURED_DIRECTORY_REF, securedDirectoryRef, ciId)
    }
    catch {
      case e: Exception =>
        error(s"Error storing securedCi [$securedDirectoryRef]: ${e.getMessage}")
        throw e
    }
}

@Repository
@Transactional(value = "reportingTransactionManager")
class SqlTaskArchive @Autowired()(@Qualifier("reportingDataSource") val dataSource: DataSource,
                                  @Qualifier("reportingSchema") implicit val schemaInfo: SchemaInfo,
                                  @Autowired @Qualifier("reportingTransactionManager") val reportingTransactionManager: PlatformTransactionManager,
                                  @Autowired roleService: RoleService,
                                  @Autowired referencedPermissionServiceClient: ReferencedPermissionServiceClient) extends TaskArchive with SqlTaskArchiveQueries with Logging {

  private[this] val template = new JdbcTemplateWithStreamingQuery(dataSource)

  override def getTask(taskId: String): TaskReader = {
    debug(s"Get task details from archive tasks for taskId: $taskId ")
    val transactionStatus = reportingTransactionManager.getTransaction(new DefaultTransactionDefinition())
    try {
      template.queryForObject(SELECT_TASK_BY_ID,
        (rs: ResultSet, _: Int) => new CachingTaskReader(rs.getCharacterStream(1)),
        taskId
      )
    } catch {
      case _: EmptyResultDataAccessException =>
        error(s"EmptyResultDataAccessException, Task Not Found in Archived Tasks for $taskId")
        transactionStatus.setRollbackOnly()
        throw new TaskNotFoundException("archive", taskId)
      case t: Throwable =>
        error(s"Exception $t for $taskId, ${ExceptionUtils.getStackTrace(t)}")
        transactionStatus.setRollbackOnly()
        throw t
    } finally {
      if (transactionStatus.isRollbackOnly) {
        reportingTransactionManager.rollback(transactionStatus)
      } else {
        reportingTransactionManager.commit(transactionStatus)
      }
    }
  }

  override def purgeTask(taskId: String): Unit = {
    if (template.update(DELETE_TASK, taskId) != 1)
      throw new IllegalArgumentException(s"Couldn't find task to purge [$taskId].")
  }

  override def searchForTasks(params: ArchivedTaskSearchParameters, streaming: Boolean): util.stream.Stream[TaskReader] = {
    /*
      TaskReader stores the stream from the ResultSet to improve performance.
      ResultSet streams will be closed when the transaction is closed.
      From https://docs.oracle.com/javase/7/docs/api/java/sql/ResultSet.html:
      Note: All the data in the returned stream must be read prior to getting the value of any other column.
        The next call to a getter method implicitly closes the stream.
        Also, a stream may return 0 when the method InputStream.available is called, whether there is data available or not.
     */
    def extractTask(rs: ResultSet) = if (streaming) new StreamingTaskReader(rs.getCharacterStream(1)) else new CachingTaskReader(rs.getCharacterStream(1))

    val selectBuilder = new SqlArchivedTaskSelectQuery(params, Seq(ArchivedTasksShared.task_id)).selectBuilder
    handleStreaming(streaming, template.queryStream(
      selectTaskDetails(selectBuilder),
      Setter(selectBuilder.parameters, Setter.setStringForReporting)
    ).map(extractTask))
  }

  override def searchForMaps(params: ArchivedTaskSearchParameters, streaming: Boolean): util.stream.Stream[util.Map[String, Object]] = {
    val finalBuilder = if (params.hasSecurityParameters) {
      createWithPermissionsQuery(params)
    }
    else {
      createQuery(params)
    }

    handleStreaming(streaming, template.queryStream(finalBuilder.query, Setter(finalBuilder.parameters, Setter.setStringForReporting)).map(toMap))
  }

  private def createQuery(params: ArchivedTaskSearchParameters) = {
    new SqlArchivedTaskSelectQuery(params).selectBuilder match {
      case selectBuilder: SelectBuilder =>
        Option(params.getFetchMode).getOrElse(FetchMode.SUMMARY) match {
          case FetchMode.SUMMARY => selectBuilder
          case FetchMode.FULL => createFullFetchModeQuery(new JoinBuilder(selectBuilder.as("summary")))
        }
      case other => other
    }
  }

  private def createWithPermissionsQuery(params: ArchivedTaskSearchParameters) = {
    new SqlArchivedTaskSelectWithPermissionsQuery(params).selectBuilder match {
      case joinBuilder: JoinBuilder =>
        Option(params.getFetchMode).getOrElse(FetchMode.SUMMARY) match {
          case FetchMode.SUMMARY => joinBuilder
          case FetchMode.FULL => createFullFetchModeQuery(joinBuilder)
        }
      case other => other
    }
  }

  private def createFullFetchModeQuery(joinBuilder: JoinBuilder): JoinBuilder = {
    joinBuilder
      .join(
        new SelectBuilder(ArchivedTasks.tableName).as("task"),
        cond.equals(ArchivedDeploymentTasks.task_id.tableAlias("summary"), ArchivedTasks.task_id.tableAlias("task")),
        JoinType.Left
      )
  }

  private def handleStreaming[T](streaming: Boolean, result: util.stream.Stream[T]) = {
    if (!streaming)
      result.collect(util.stream.Collectors.toList()).stream()
    else
      result
  }

  private def getDeploymentTaskAnalyticsQuery(begin: DateTime, end: DateTime): DeploymentTaskAnalyticsQuery = {
    if (!PermissionHelper.isCurrentUserAdmin && !roleService.isReadOnlyAdmin()) {
      new SqlDeploymentTaskAnalyticsWithPermissionsQuery(begin, end, getReferences.asScala.toList)
    } else {
      new SqlDeploymentTaskAnalyticsQuery(begin, end)
    }
  }

  override def getReferences: util.List[String] ={
    val rolesPermissionsPair = PlatformPermissions.READ.getPermissionHandler.getRolesPermissionsPair
    referencedPermissionServiceClient.getReferencesForRoles(
      rolesPermissionsPair.getRoles.asScala.toList.map(UUID.fromString),
      rolesPermissionsPair.getPermissions.asScala.toList
    ).map(_.toString).toList.asJava
  }

  override def topNDeploymentsByState(begin: DateTime, end: DateTime, deploymentResult: DeploymentResult, limit: Int): util.Collection[TopNByStateReportLine] = {
    val mainQuery = getDeploymentTaskAnalyticsQuery(begin, end)
    val selectBuilder = mainQuery.topNDeploymentsByState(deploymentResult, limit)

    template.query(selectBuilder.query, Setter(selectBuilder.parameters, Setter.setStringForReporting), (rs: ResultSet, rowNum: Int) => {
      val applicationName = rs.getString(1)
      val environment = rs.getString(2)
      val subQuery = getDeploymentTaskAnalyticsQuery(begin, end)
      val totalBuilder = subQuery.totalsByAppAndEnv(applicationName, environment)
      val totalDeployTasks = template.query(totalBuilder.query, Setter(totalBuilder.parameters, Setter.setStringForReporting), new SingleColumnRowMapper[Number]).get(0).intValue()
      TopNByStateReportLine(applicationName, environment, rs.getInt(3), totalDeployTasks)
    })
  }

  override def statusOverview(begin: DateTime, end: DateTime): util.Collection[StatusOverviewReportLine] = {
    val rows = new util.ArrayList[StatusOverviewReportLine]()

    DeploymentResult.values().foreach(status => {
      val query = getDeploymentTaskAnalyticsQuery(begin, end)
      val builder = query.countByDeploymentResult(status)

      template.query(builder.query, Setter(builder.parameters, Setter.setStringForReporting), (rs: ResultSet, rowNum: Int) => {
        rows.add(StatusOverviewReportLine(status.toString, rs.getInt(1)))
      })
    })

    rows
  }

  override def throughput(begin: DateTime, end: DateTime): util.Collection[ThroughputReportLine] = {
    val rows = new util.ArrayList[ThroughputReportLine]()

    DeploymentResult.values().foreach(status => {
      val query = getDeploymentTaskAnalyticsQuery(begin, end)
      val builder = query.throughputByDeploymentResult(status)

      template.query(builder.query, Setter(builder.parameters, Setter.setStringForReporting), (rs: ResultSet, rowNum: Int) => {
        rows.add(ThroughputReportLine(rs.getString(1).toDouble.toInt, rs.getString(2).toDouble.toInt, rs.getInt(3), status.toString))
      })
    })

    rows
  }

  override def appEnvStatus(begin: DateTime, end: DateTime): util.Collection[AppEnvStatusReportLine] = {
    val rows = new util.ArrayList[AppEnvStatusReportLine]()

    DeploymentResult.values().foreach(status => {
      val query = getDeploymentTaskAnalyticsQuery(begin, end)
      val builder = query.appEnvStatusByDeploymentResult(status)

      template.query(builder.query, Setter(builder.parameters, Setter.setStringForReporting), (rs: ResultSet, rowNum: Int) => {
        rows.add(AppEnvStatusReportLine(rs.getString(1), rs.getString(2), rs.getInt(3), status.toString))
      })
    })
    rows
  }

  @SuppressWarnings(scala.Array("all")) // FIXME: Use of Traversable.head is considered unsafe
  override def successfulDeploymentDurations(begin: DateTime, end: DateTime): util.Collection[SuccessfulDeploymentDurationReportLine] = {
    val mainQuery = getDeploymentTaskAnalyticsQuery(begin, end)
    val mainBuilder = mainQuery.totalSuccessfulDeploymentDuration()

    val (count: Int, mean: Double, stddev: Double) = template.query(mainBuilder.query, Setter(mainBuilder.parameters, Setter.setStringForReporting), (rs: ResultSet, rowNum: Int) => {
      (rs.getInt(1), rs.getDouble(2), rs.getDouble(3))
    }).asScala.head

    val rows = new util.ArrayList[SuccessfulDeploymentDurationReportLine]()
    bins(mean, stddev).foreach {
      case (binStart, binEnd) =>
        val subQuery = getDeploymentTaskAnalyticsQuery(begin, end)
        val subBuilder = subQuery.bins(binStart, binEnd)

        template.query(subBuilder.query, Setter(subBuilder.parameters, Setter.setStringForReporting), (rs: ResultSet, _: Int) => {
          rows.add(SuccessfulDeploymentDurationReportLine(rs.getInt(1), count, binStart, binEnd))
        })
    }

    rows
  }

  override def topNLongestRunningDeployments(begin: DateTime, end: DateTime, topN: Int): util.Collection[TopNLongestRunningReportLine] = {
    val query = getDeploymentTaskAnalyticsQuery(begin, end)
    val builder = query.topNLongestRunningDeployments(topN)

    template.query(builder.query, Setter(builder.parameters, Setter.setStringForReporting), (rs: ResultSet, rowNum: Int) =>
      TopNLongestRunningReportLine(rs.getLong(1), rs.getString(2), rs.getString(3))
    )
  }

  override def averageDurationDeploymentsOverTime(begin: DateTime, end: DateTime): util.Collection[AverageDeploymentsDurationOverTimeReportLine] = {
    val query = getDeploymentTaskAnalyticsQuery(begin, end)
    val builder = query.averageDurationDeploymentsOverTime()

    template.query(builder.query, Setter(builder.parameters, Setter.setStringForReporting), (rs: ResultSet, rowNum: Int) =>
      AverageDeploymentsDurationOverTimeReportLine(rs.getString(1).toInt, rs.getString(2).toInt, rs.getInt(3), rs.getLong(4))
    )
  }

  override def getAllArchivedEnvironments: util.List[String] = {
    val params = new ArchivedTaskSearchParameters().thatAreOfType(DeploymentTask.DEPLOYMENT_TASK_TYPES)
    params.orderBy("environment_id", true)
    val selectBuilder = new SqlArchivedTaskSelectQuery(params, Seq(new StaticSqlFunction("distinct %s")(ArchivedDeploymentTasks.environment)))
      .selectBuilder
    template.query[String](selectBuilder.query, Setter(selectBuilder.parameters, Setter.setStringForReporting), (rs: ResultSet, rownum: Int) => rs.getString(1))
  }


  override def getAllArchivedEnvironmentsWithIds: util.Map[Integer, util.List[String]] = {
    val params = new ArchivedTaskSearchParameters().thatAreOfType(DeploymentTask.DEPLOYMENT_TASK_TYPES)
    params.orderBy("environment_id", true)
    val selectBuilder = new SqlArchivedTaskSelectQuery(params,
      Seq(new StaticSqlFunction("distinct %s, %s")(ArchivedDeploymentTasks.environment_internal_id, ArchivedDeploymentTasks.environment)))
      .selectBuilder

    queryForEntities(selectBuilder.query, selectBuilder.parameters)
  }

  override def getAllArchivedApplicationsWithIds: util.Map[Integer, util.List[String]] = {
    val builder = new SelectBuilder(ArchivedDeploymentTasks.Applications.tableName)
      .select(ArchivedDeploymentTasks.Applications.application_internal_id)
      .select(ArchivedDeploymentTasks.Applications.application)
      .distinct()

    queryForEntities(builder.query, builder.parameters)
  }

  def queryForEntities(query: String, parameters: Seq[Any]): util.Map[Integer, util.List[String]] = {
    template.query(query, Setter(parameters, Setter.setStringForReporting), new RowMapper[(Integer, String)] {
      override def mapRow(rs: ResultSet, rowNum: Int): (Integer, String) = {
        (rs.getInt(1), rs.getString(2))
      }
    }).asScala.groupBy(_._1).mapValues(_.map(_._2).toList.asJava).toMap.asJava
  }

  override def lastDeploymentsOnEnvironment(environment: String, before: DateTime, count: Int): stream.Stream[util.Map[String, AnyRef]] = {
    val selectBuilder = new TopNPerGroupSelectBuilder(ArchivedDeploymentTasks.tableName, count, ArchivedDeploymentTasks.task_id)
      .partitionBy(ArchivedDeploymentTasks.main_application)
      .where(cond.equals(ArchivedDeploymentTasks.environment, environment))
      .where(cond.equals(ArchivedDeploymentTasks.status, TaskExecutionState.DONE.name()))
      .where(cond.before(ArchivedDeploymentTasks.end_date, before))
      .orderBy(OrderBy.desc(ArchivedDeploymentTasks.end_date))
    template.queryStream(selectBuilder.query, Setter(selectBuilder.parameters, Setter.setStringForReporting)).map(toMap)
  }

  override def countSuccessfulDeploymentsOfApplications(appIds: util.List[String]): util.Map[String, Integer] = {
    val subbuilder = new SelectBuilder(ArchivedDeploymentTasks.Applications.tableName)
      .select(ArchivedDeploymentTasks.Applications.task_id)
      .where(cond.in(ArchivedDeploymentTasks.Applications.application, appIds.asScala))
    val builder = new SelectBuilder(ArchivedDeploymentTasks.tableName)
      .select(ArchivedDeploymentTasks.main_application)
      .select(func.countAll)
      .where(cond.equals(ArchivedDeploymentTasks.status, TaskExecutionState.DONE.name()))
      .where(cond.subselect(ArchivedDeploymentTasks.task_id, subbuilder))
      .groupBy(ArchivedDeploymentTasks.main_application)
    template.query(builder.query, Setter(builder.parameters, Setter.setStringForReporting), (rs: ResultSet) => {
      val result = new util.HashMap[String, Integer]()
      while (rs.next()) {
        result.put(rs.getString(1), rs.getInt(2))
      }
      result
    })
  }

  override def countSuccessfulDeploymentsOnEnvironments(envIds: util.List[String]): util.Map[String, Integer] = {
    val builder = new SelectBuilder(ArchivedDeploymentTasks.tableName)
      .select(ArchivedDeploymentTasks.environment)
      .select(func.countAll)
      .where(cond.in(ArchivedDeploymentTasks.environment, envIds.asScala))
      .where(cond.equals(ArchivedDeploymentTasks.status, TaskExecutionState.DONE.name()))
      .groupBy(ArchivedDeploymentTasks.environment)
    template.query(builder.query, Setter(builder.parameters, Setter.setStringForReporting), (rs: ResultSet) => {
      val result = new util.HashMap[String, Integer]()
      while (rs.next()) {
        result.put(rs.getString(1), rs.getInt(2))
      }
      result
    })
  }

  override def countTotalResults(params: ArchivedTaskSearchParameters): Int = {
    val selectBuilder =
      if (params.hasSecurityParameters) {
        new SqlArchivedTaskCountWithPermissionsQuery(params).selectBuilder
      }
      else {
        new SqlArchivedTaskCountQuery(params).selectBuilder
      }
    template.query[Int](selectBuilder.query, Setter(selectBuilder.parameters, Setter.setStringForReporting), { (rs: ResultSet, rn: Int) =>
      rs.getInt(1)
    }).asScala.sum
  }

  override def markTaskAsRolledBack(taskId: String): Unit = {
    if (template.update(UPDATE_ROLLBACK_STATUS, TRUE, taskId) != 1)
      throw new IllegalArgumentException(s"Couldn't update rollback status for: [$taskId].")
  }
}

trait SqlTaskArchiveQueries extends Queries {

  lazy val SELECT_TASK_BY_ID: String = {
    import com.xebialabs.deployit.task.archive.sql.schema.ArchivedTasks._
    sqlb"select $task_details from $tableName where $task_id = ?"
  }

  lazy val DELETE_TASK: String = {
    import com.xebialabs.deployit.task.archive.sql.schema.ArchivedTasks._
    sqlb"delete from $tableName where $task_id = ?"
  }

  lazy val UPDATE_ROLLBACK_STATUS: String = {
    import com.xebialabs.deployit.task.archive.sql.schema.ArchivedDeploymentTasks._
    sqlb"update $tableName set $rolled_back = ? where $task_id = ?"
  }

  lazy val UPDATE_ENVIRONMENT_SECURE_CI: String = {
    import com.xebialabs.deployit.task.archive.sql.schema.ArchivedDeploymentTasks._
    sqlb"update $tableName set $environment_secured_ci = ? where $environment_internal_id = ?"
  }

  lazy val UPDATE_APPLICATION_SECURE_CI: String = {
    import com.xebialabs.deployit.task.archive.sql.schema.ArchivedDeploymentTasks.Applications._
    sqlb"update $tableName set $application_secured_ci = ? where $application_internal_id = ?"
  }

  lazy val UPDATE_CONTROL_TASK_SECURE_CI: String = {
    import com.xebialabs.deployit.task.archive.sql.schema.ArchivedControlTasks._
    sqlb"update $tableName set $target_secured_ci = ? where $target_internal_id = ?"
  }

  lazy val UPDATE_ENVIRONMENT_SECURED_DIRECTORY_REF: String = {
    import com.xebialabs.deployit.task.archive.sql.schema.ArchivedDeploymentTasks._
    sqlb"update $tableName set $environment_dir_ref = ? where $environment_internal_id = ?"
  }

  lazy val UPDATE_APPLICATION_SECURED_DIRECTORY_REF: String = {
    import com.xebialabs.deployit.task.archive.sql.schema.ArchivedDeploymentTasks.Applications._
    sqlb"update $tableName set $application_dir_ref = ? where $application_internal_id = ?"
  }

  lazy val UPDATE_CONTROL_TASK_SECURED_DIRECTORY_REF: String = {
    import com.xebialabs.deployit.task.archive.sql.schema.ArchivedControlTasks._
    sqlb"update $tableName set $target_dir_ref = ? where $target_internal_id = ?"
  }


  def selectTaskDetails(selectBuilder: AbstractQueryBuilder): String = {
    import com.xebialabs.deployit.task.archive.sql.schema.ArchivedTasks._
    val orderBy = selectBuilder.getOrderBy
    selectBuilder.clearOrderBy()
    val page = selectBuilder.getPage
    selectBuilder.clearPage()
    val query = sqlb"select $task_details from $tableName tasks inner join (${selectBuilder.query}) selected on (tasks.$task_id = selected.$task_id)${orderByClause(orderBy, Some("selected"))}"
    page.map(p => schemaInfo.sqlDialect.addPaging(query, p.page, p.size)).getOrElse(query)
  }

  def getSecureCiSetter(securedCi: Integer, ciIdList: List[Integer]): BatchPreparedStatementSetter = {
    new BatchPreparedStatementSetter {
      override def getBatchSize: Int = ciIdList.size

      override def setValues(ps: PreparedStatement, i: Int): Unit = {
        ps.setInt(1, securedCi)
        ps.setInt(2, ciIdList(i))
      }
    }
  }

  def getSecuredDirectoryReferenceSetter(securedDirectoryRef: String, ciIdList: List[Integer]): BatchPreparedStatementSetter = {
    new BatchPreparedStatementSetter {
      override def getBatchSize: Int = ciIdList.size

      override def setValues(ps: PreparedStatement, i: Int): Unit = {
        ps.setString(1, securedDirectoryRef)
        ps.setInt(2, ciIdList(i))
      }
    }
  }
}

private[this] class JdbcTemplateWithStreamingQuery(dataSource: DataSource) extends JdbcTemplate(dataSource) {

  //Copied from org.springframework.jdbc.core.JdbcTemplate.execute(org.springframework.jdbc.core.PreparedStatementCreator, org.springframework.jdbc.core.PreparedStatementCallback<T>)
  def queryStream(sql: String, pss: PreparedStatementSetter): util.stream.Stream[ResultSet] = {
    if (logger.isDebugEnabled) {
      logger.debug("Executing streaming prepared SQL statement [" + sql + "]")
    }

    var con = DataSourceUtils.getConnection(getDataSource)
    var ps: PreparedStatement = null
    try {
      ps = con.prepareStatement(sql, ResultSet.TYPE_FORWARD_ONLY, ResultSet.CONCUR_READ_ONLY)
      ps.setFetchSize(100)
      applyStatementSettings(ps)
      pss.setValues(ps)
      val result = StreamingResultSet.from(ps.executeQuery())
      handleWarnings(ps)
      result
    } catch {
      case ex: SQLException =>
        JdbcUtils.closeStatement(ps)
        ps = null
        DataSourceUtils.releaseConnection(con, getDataSource)
        con = null
        throw getExceptionTranslator.translate("PreparedStatementCallback", sql, ex)
    }

    // Do not close ps or rs in finally.
  }
  override protected def newArgPreparedStatementSetter(@Nullable args: scala.Array[AnyRef]) =
    new ExtendedArgumentPreparedStatementSetter(args, false)


}
