package com.xebialabs.deployit.task.archive.sql

import com.xebialabs.deployit.core.sql.{SqlCondition => cond, _}
import com.xebialabs.deployit.security.archive.sql.schema.ArchivePermissionsSchema
import com.xebialabs.deployit.task.archive.sql.schema.{ArchivedControlTasks, ArchivedDeploymentTasks}

class ArchiveTaskSelectWithPermissionQuery(val roles: List[String],
                                           val permissions: List[String])(implicit val schemaInfo: SchemaInfo) {

  import ArchiveTaskSelectWithPermissionQuery._

  def applicationTaskQuery: SelectBuilder = new SelectBuilder(ArchivedDeploymentTasks.Applications.tableName).as(aliasArchivedApplications)

  def deploymentTaskQuery(baseTaskSelectBuilder: SelectBuilder, applicationSelectBuilder: SelectBuilder = applicationTaskQuery): JoinBuilder =
    new JoinBuilder(baseTaskSelectBuilder.as(aliasArchivedDeploymentTasks)).join(
      applicationSelectBuilder,
      cond.equals(
        ArchivedDeploymentTasks.Applications.task_id.tableAlias(aliasArchivedApplications),
        ArchivedDeploymentTasks.task_id.tableAlias(aliasArchivedDeploymentTasks))
    )
      .join(
        permissionsQuery(aliasArchivedDeploymentTaskPermissions),
        permissionCondition(aliasArchivedDeploymentTaskPermissions, ArchivedDeploymentTasks.environment_secured_ci.tableAlias(aliasArchivedDeploymentTasks))
      )
      .join(
        permissionsQuery(aliasArchivedApplicationPermissions),
        permissionCondition(aliasArchivedApplicationPermissions, ArchivedDeploymentTasks.Applications.application_secured_ci.tableAlias(aliasArchivedApplications))
      )

  def controlTaskQuery(baseTaskSelectBuilder: SelectBuilder): AbstractQueryBuilder =
    new JoinBuilder(baseTaskSelectBuilder.as(aliasControlTasks))
      .join(
        permissionsQuery(aliasControlTaskPermissions),
        permissionCondition(aliasControlTaskPermissions,
          ArchivedControlTasks.target_secured_ci.tableAlias(aliasControlTasks))
      )

  private def permissionCondition(subqueryAlias: String, joinColumn: Selectable): cond = {
    cond.joinSubselect(cond.equals(ArchivePermissionsSchema.CI_ID.tableAlias(subqueryAlias), joinColumn))
  }

  private def permissionsQuery(tableAlias: String): SelectBuilder = {
    new SelectBuilder(ArchivePermissionsSchema.tableName).as(tableAlias)
      .select(ArchivePermissionsSchema.CI_ID).distinct()
      .where(cond.and(Seq(
        cond.in(ArchivePermissionsSchema.ROLE_ID, roles),
        cond.in(ArchivePermissionsSchema.PERMISSION_NAME, permissions)
      )))
  }
}

object ArchiveTaskSelectWithPermissionQuery {
  val aliasControlTasks = "act"
  val aliasArchivedDeploymentTasks = "adt"
  val aliasArchivedApplications = "adtapp"

  val aliasArchivedDeploymentTaskPermissions = "adtperm"
  val aliasArchivedApplicationPermissions = "adtappperm"
  val aliasArchivedDeploymentTaskRoles = "adtrole"
  val aliasArchivedApplicationRoles = "adtapprole"

  val aliasControlTaskPermissions = "actperm"
  val aliasControlTaskRoles = "acttrole"
}
