package com.xebialabs.deployit.task.archive.sql

import com.xebialabs.deployit.core.sql._

class TopNPerGroupSelectBuilder(table: TableName, number: Int, primaryKey: ColumnName)(implicit override val schemaInfo: SchemaInfo) extends SelectBuilder(table) with Queries {

  private var partitionBy: Seq[Selectable] = Seq()

  def partitionBy(column: Selectable): TopNPerGroupSelectBuilder = {
    partitionBy = partitionBy :+ column
    this
  }

  private[this] val delegate =
    if (schemaInfo.sqlDialect.supportPartitionBy)
      new PartitionTopNPerGroup
    else
      new JoinTopNPerGroup

  private[this] def superParameters = super.parameters

  class PartitionTopNPerGroup extends QueryBuilder {
    private[this] def partitionExpression:String = partitionBy.map(_.build(None)).mkString(", ")

    override def query: String = sqlb"select * from (select a.$selectFragment, row_number() over (partition by $partitionExpression$orderByFragment) as rn from $table a$whereFragment) x where x.rn < ?"
    override def parameters: Seq[Any] = superParameters :+ (number+1)
  }

  class JoinTopNPerGroup extends QueryBuilder {
    private[this] def orderByColumns = orderBy.map(_.column)
    private[this] def columnConditions = where.filter(_.isInstanceOf[ConditionOnColumn]).map(_.asInstanceOf[ConditionOnColumn])
    private[this] def orderByConditions = columnConditions.filter(c => orderByColumns.contains(c.column))
    private[this] def otherConditions: Seq[SqlCondition] = where.diff(orderByConditions)

    override def query: String = {
      val joinCondition =
        ((partitionBy ++ columnConditions.map(_.column).filterNot { it: Selectable => orderByColumns.contains(it) }).map(column => s"${column.build(Some("x3"))} = ${column.build(Some("x4"))}") ++
          orderBy.map {
            case AscOrdering(column) => s"${column.build(Some("x3"))} >= ${column.build(Some("x4"))}"
            case DescOrdering(column) => s"${column.build(Some("x3"))} <= ${column.build(Some("x4"))}"
          }).mkString(" and ")
      val allConditions = otherConditions.map(_.build(Some("x3"))) ++ orderByConditions.map(_.build(Some("x3"))) ++ orderByConditions.map(_.build(Some("x4")))
      val allConditionExpression = mkString(allConditions, "", " where ", " and ", "")
      sqlb"select x1.*, x2.rn from $table x1 join (select x3.$primaryKey, count(*) as rn from $table x3 join $table x4 on ($joinCondition)$allConditionExpression group by x3.$primaryKey having count(*) < ?) x2 on (x1.$primaryKey = x2.$primaryKey)"
    }
    override def parameters: Seq[Any] =
      (otherConditions ++ orderByConditions.flatMap(c => Seq(c, c))).flatMap(_.arguments) :+ (number+1)
  }

  override def query: String = delegate.query
  override def parameters: Seq[Any] = delegate.parameters
}
